//===- SPIRVToLLVMPass.cpp - SPIR-V to LLVM Passes ------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// This file implements a pass to convert MLIR SPIR-V ops into LLVM ops
//
//===----------------------------------------------------------------------===//

#include "mlir/Conversion/SPIRVToLLVM/SPIRVToLLVMPass.h"
#include "../PassDetail.h"
#include "mlir/Conversion/SPIRVToLLVM/SPIRVToLLVM.h"
#include "mlir/Conversion/StandardToLLVM/ConvertStandardToLLVM.h"
#include "mlir/Conversion/StandardToLLVM/ConvertStandardToLLVMPass.h"
#include "mlir/Dialect/LLVMIR/LLVMDialect.h"
#include "mlir/Dialect/SPIRV/IR/SPIRVDialect.h"

using namespace mlir;

namespace {
/// A pass converting MLIR SPIR-V operations into LLVM dialect.
class ConvertSPIRVToLLVMPass
    : public ConvertSPIRVToLLVMBase<ConvertSPIRVToLLVMPass> {
  void runOnOperation() override;
};
} // namespace

void ConvertSPIRVToLLVMPass::runOnOperation() {
  MLIRContext *context = &getContext();
  ModuleOp module = getOperation();
  LLVMTypeConverter converter(&getContext());

  // Encode global variable's descriptor set and binding if they exist.
  encodeBindAttribute(module);

  OwningRewritePatternList patterns;

  populateSPIRVToLLVMTypeConversion(converter);

  populateSPIRVToLLVMModuleConversionPatterns(context, converter, patterns);
  populateSPIRVToLLVMConversionPatterns(context, converter, patterns);
  populateSPIRVToLLVMFunctionConversionPatterns(context, converter, patterns);

  ConversionTarget target(getContext());
  target.addIllegalDialect<spirv::SPIRVDialect>();
  target.addLegalDialect<LLVM::LLVMDialect>();

  // Set `ModuleOp` and `ModuleTerminatorOp` as legal for `spv.module`
  // conversion.
  target.addLegalOp<ModuleOp>();
  target.addLegalOp<ModuleTerminatorOp>();
  if (failed(applyPartialConversion(module, target, std::move(patterns))))
    signalPassFailure();
}

std::unique_ptr<OperationPass<ModuleOp>> mlir::createConvertSPIRVToLLVMPass() {
  return std::make_unique<ConvertSPIRVToLLVMPass>();
}
