// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using Test.Cryptography;

namespace System.Security.Cryptography.Pkcs.Tests
{
    internal static class SignedDocuments
    {
        internal static readonly byte[] RsaPssDocument = (
            "308204EC06092A864886F70D010702A08204DD308204D9020103310D300B0609" +
            "608648016503040201301F06092A864886F70D010701A0120410546869732069" +
            "73206120746573740D0AA08202CA308202C63082022EA003020102020900F399" +
            "4D1706DEC3F8300D06092A864886F70D01010B0500307B310B30090603550406" +
            "130255533113301106035504080C0A57617368696E67746F6E3110300E060355" +
            "04070C075265646D6F6E6431183016060355040A0C0F4D6963726F736F667420" +
            "436F72702E31173015060355040B0C0E2E4E4554204672616D65776F726B3112" +
            "301006035504030C096C6F63616C686F7374301E170D31363033303230323337" +
            "35345A170D3137303330323032333735345A307B310B30090603550406130255" +
            "533113301106035504080C0A57617368696E67746F6E3110300E06035504070C" +
            "075265646D6F6E6431183016060355040A0C0F4D6963726F736F667420436F72" +
            "702E31173015060355040B0C0E2E4E4554204672616D65776F726B3112301006" +
            "035504030C096C6F63616C686F73743081A0300D06092A864886F70D01010105" +
            "0003818E0030818A02818200BCACB1A5349D7B35A580AC3B3998EB15EBF900EC" +
            "B329BF1F75717A00B2199C8A18D791B592B7EC52BD5AF2DB0D3B635F0595753D" +
            "FF7BA7C9872DBF7E3226DEF44A07CA568D1017992C2B41BFE5EC3570824CF1F4" +
            "B15919FED513FDA56204AF2034A2D08FF04C2CCA49D168FA03FA2FA32FCCD348" +
            "4C15F0A2E5467C76FC760B55090203010001A350304E301D0603551D0E041604" +
            "141063CAB14FB14C47DC211C0E0285F3EE5946BF2D301F0603551D2304183016" +
            "80141063CAB14FB14C47DC211C0E0285F3EE5946BF2D300C0603551D13040530" +
            "030101FF300D06092A864886F70D01010B050003818200435774FB66802AB3CE" +
            "2F1392C079483B48CC8913E0BF3B7AD88351E4C15B55CAD3061AA5875900C56B" +
            "2E7E84BB49CA2A0C1895BD60149C6A0AE983E48370E2144052943B066BD85F70" +
            "543CF6F2F255C028AE1DC8FB898AD3DCA97BF1D607370287077A4C147268C911" +
            "8CF9CAD318D2830D3468727E0A3247B3FEB8D87A7DE4F1E2318201D4308201D0" +
            "02010380141063CAB14FB14C47DC211C0E0285F3EE5946BF2D300B0609608648" +
            "016503040201A081E4301806092A864886F70D010903310B06092A864886F70D" +
            "010701301C06092A864886F70D010905310F170D313731303236303130363235" +
            "5A302F06092A864886F70D0109043122042007849DC26FCBB2F3BD5F57BDF214" +
            "BAE374575F1BD4E6816482324799417CB379307906092A864886F70D01090F31" +
            "6C306A300B060960864801650304012A300B0609608648016503040116300B06" +
            "09608648016503040102300A06082A864886F70D0307300E06082A864886F70D" +
            "030202020080300D06082A864886F70D0302020140300706052B0E030207300D" +
            "06082A864886F70D0302020128303D06092A864886F70D01010A3030A00D300B" +
            "0609608648016503040201A11A301806092A864886F70D010108300B06096086" +
            "48016503040201A20302015F048181B93E81D141B3C9F159AB0021910635DC72" +
            "E8E860BE43C28E5D53243D6DC247B7D4F18C20195E80DEDCC75B29C43CE5047A" +
            "D775B65BFC93589BD748B950C68BADDF1A4673130302BBDA8667D5DDE5EA91EC" +
            "CB13A9B4C04F1C4842FEB1697B7669C7692DD3BDAE13B5AA8EE3EB5679F3729D" +
            "1DC4F2EB9DC89B7E8773F2F8C6108C05").HexToByteArray();

        public static byte[] RsaPkcs1OneSignerIssuerAndSerialNumber = (
            "3082033706092A864886F70D010702A082032830820324020101310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08202103082020C30820179A0030201020210" +
            "5D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500301E311C30" +
            "1A060355040313135253414B65795472616E736665724361706931301E170D31" +
            "35303431353037303030305A170D3235303431353037303030305A301E311C30" +
            "1A060355040313135253414B65795472616E73666572436170693130819F300D" +
            "06092A864886F70D010101050003818D0030818902818100AA272700586C0CC4" +
            "1B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA9596C5C63BA2B1" +
            "AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44EC86AA8F6C9D2" +
            "B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37DA4093EF76C1" +
            "3B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203010001A35330" +
            "51304F0603551D0104483046801015432DB116B35D07E4BA89EDB2469D7AA120" +
            "301E311C301A060355040313135253414B65795472616E736665724361706931" +
            "82105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500038181" +
            "0081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF23369FA533C8DC19" +
            "38952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70D54646975B6D" +
            "C2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653B5C78E5291E4" +
            "401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F291822E99A4296" +
            "463181D73081D40201013032301E311C301A060355040313135253414B657954" +
            "72616E73666572436170693102105D2FFFF863BABC9B4D3C80AB178A4CCA3009" +
            "06052B0E03021A0500300D06092A864886F70D01010105000481805A1717621D" +
            "450130B3463662160EEC06F7AE77E017DD95F294E97A0BDD433FE6B2CCB34FAA" +
            "C33AEA50BFD7D9E78DC7174836284619F744278AE77B8495091E096EEF682D9C" +
            "A95F6E81C7DDCEDDA6A12316B453C894B5000701EB09DF57A53B733A4E80DA27" +
            "FA710870BD88C86E2FDB9DCA14D18BEB2F0C87E9632ABF02BE2FE3").HexToByteArray();

        public static byte[] CounterSignedRsaPkcs1OneSigner = (
            "3082044906092A864886F70D010702A082043A30820436020101310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08202103082020C30820179A0030201020210" +
            "5D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500301E311C30" +
            "1A060355040313135253414B65795472616E736665724361706931301E170D31" +
            "35303431353037303030305A170D3235303431353037303030305A301E311C30" +
            "1A060355040313135253414B65795472616E73666572436170693130819F300D" +
            "06092A864886F70D010101050003818D0030818902818100AA272700586C0CC4" +
            "1B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA9596C5C63BA2B1" +
            "AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44EC86AA8F6C9D2" +
            "B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37DA4093EF76C1" +
            "3B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203010001A35330" +
            "51304F0603551D0104483046801015432DB116B35D07E4BA89EDB2469D7AA120" +
            "301E311C301A060355040313135253414B65795472616E736665724361706931" +
            "82105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500038181" +
            "0081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF23369FA533C8DC19" +
            "38952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70D54646975B6D" +
            "C2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653B5C78E5291E4" +
            "401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F291822E99A4296" +
            "46318201E8308201E40201013032301E311C301A060355040313135253414B65" +
            "795472616E73666572436170693102105D2FFFF863BABC9B4D3C80AB178A4CCA" +
            "300906052B0E03021A0500300D06092A864886F70D01010105000481805A1717" +
            "621D450130B3463662160EEC06F7AE77E017DD95F294E97A0BDD433FE6B2CCB3" +
            "4FAAC33AEA50BFD7D9E78DC7174836284619F744278AE77B8495091E096EEF68" +
            "2D9CA95F6E81C7DDCEDDA6A12316B453C894B5000701EB09DF57A53B733A4E80" +
            "DA27FA710870BD88C86E2FDB9DCA14D18BEB2F0C87E9632ABF02BE2FE3A18201" +
            "0C3082010806092A864886F70D0109063181FA3081F702010380146B4A6B92FD" +
            "ED07EE0119F3674A96D1A70D2A588D300906052B0E03021A0500A03F30180609" +
            "2A864886F70D010903310B06092A864886F70D010701302306092A864886F70D" +
            "010904311604148C054D6DF2B08E69A86D8DB23C1A509123F9DBA4300D06092A" +
            "864886F70D0101010500048180962518DEF789B0886C7E6295754ECDBDC4CB9D" +
            "153ECE5EBBE7A82142B92C30DDBBDFC22B5B954F5D844CBAEDCA9C4A068B2483" +
            "0E2A96141A5D0320B69EA5DFCFEA441E162D04506F8FFA79D7312524F111A9B9" +
            "B0184007139F94E46C816E0E33F010AEB949F5D884DC8987765002F7A643F34B" +
            "7654E3B2FD5FB34A420279B1EA").HexToByteArray();

        public static byte[] NoSignatureSignedWithAttributesAndCounterSignature = (
            "3082042406092A864886F70D010702A082041530820411020101310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08202103082020C30820179A0030201020210" +
            "5D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500301E311C30" +
            "1A060355040313135253414B65795472616E736665724361706931301E170D31" +
            "35303431353037303030305A170D3235303431353037303030305A301E311C30" +
            "1A060355040313135253414B65795472616E73666572436170693130819F300D" +
            "06092A864886F70D010101050003818D0030818902818100AA272700586C0CC4" +
            "1B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA9596C5C63BA2B1" +
            "AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44EC86AA8F6C9D2" +
            "B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37DA4093EF76C1" +
            "3B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203010001A35330" +
            "51304F0603551D0104483046801015432DB116B35D07E4BA89EDB2469D7AA120" +
            "301E311C301A060355040313135253414B65795472616E736665724361706931" +
            "82105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500038181" +
            "0081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF23369FA533C8DC19" +
            "38952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70D54646975B6D" +
            "C2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653B5C78E5291E4" +
            "401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F291822E99A4296" +
            "46318201C3308201BF020101301C3017311530130603550403130C44756D6D79" +
            "205369676E6572020100300906052B0E03021A0500A05D301806092A864886F7" +
            "0D010903310B06092A864886F70D010701301C06092A864886F70D010905310F" +
            "170D3137313130313137313731375A302306092A864886F70D01090431160414" +
            "A5F085E7F326F3D6CA3BFD6280A3DE8EBC2EA60E300C06082B06010505070602" +
            "050004148B70D20D0477A35CD84AB962C10DC52FBA6FAD6BA182010C30820108" +
            "06092A864886F70D0109063181FA3081F702010380146B4A6B92FDED07EE0119" +
            "F3674A96D1A70D2A588D300906052B0E03021A0500A03F301806092A864886F7" +
            "0D010903310B06092A864886F70D010701302306092A864886F70D0109043116" +
            "0414833378066BDCCBA7047EF6919843D181A57D6479300D06092A864886F70D" +
            "01010105000481802155D226DD744166E582D040E60535210195050EA00F2C17" +
            "9897198521DABD0E6B27750FD8BA5F9AAF58B4863B6226456F38553A22453CAF" +
            "0A0F106766C7AB6F3D6AFD106753DC50F8A6E4F9E5508426D236C2DBB4BCB816" +
            "2FA42E995CBA16A340FD7C793569DF1B71368E68253299BC74E38312B40B8F52" +
            "EAEDE10DF414A522").HexToByteArray();

        public static byte[] NoSignatureWithNoAttributes = (
            "30819B06092A864886F70D010702A0818D30818A020101310B300906052B0E03" +
            "021A0500302406092A864886F70D010701A01704154D6963726F736F66742043" +
            "6F72706F726174696F6E31523050020101301C3017311530130603550403130C" +
            "44756D6D79205369676E6572020100300906052B0E03021A0500300C06082B06" +
            "01050507060205000414A5F085E7F326F3D6CA3BFD6280A3DE8EBC2EA60E").HexToByteArray();

        public static byte[] RsaCapiTransfer1_NoEmbeddedCert = (
            "3082016606092A864886F70D010702A082015730820153020103310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6E318201193082011502010380146B4A6B92FDED" +
            "07EE0119F3674A96D1A70D2A588D300906052B0E03021A0500A05D301806092A" +
            "864886F70D010903310B06092A864886F70D010701301C06092A864886F70D01" +
            "0905310F170D3137313130323135333430345A302306092A864886F70D010904" +
            "31160414A5F085E7F326F3D6CA3BFD6280A3DE8EBC2EA60E300D06092A864886" +
            "F70D01010105000481800EDE3870B8A80B45A21BAEC4681D059B46502E1B1AA6" +
            "B8920CF50D4D837646A55559B4C05849126C655D95FF3C6C1B420E07DC42629F" +
            "294EE69822FEA56F32D41B824CBB6BF809B7583C27E77B7AC58DFC925B1C60EA" +
            "4A67AA84D73FC9E9191D33B36645F17FD6748A2D8B12C6C384C3C734D2727338" +
            "6211E4518FE2B4ED0147").HexToByteArray();

        public static byte[] OneRsaSignerTwoRsaCounterSigners = (
            "3082075106092A864886F70D010702A08207423082073E020101310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08203F9308201E530820152A0030201020210" +
            "D5B5BC1C458A558845BFF51CB4DFF31C300906052B0E03021D05003011310F30" +
            "0D060355040313064D794E616D65301E170D3130303430313038303030305A17" +
            "0D3131303430313038303030305A3011310F300D060355040313064D794E616D" +
            "6530819F300D06092A864886F70D010101050003818D0030818902818100B11E" +
            "30EA87424A371E30227E933CE6BE0E65FF1C189D0D888EC8FF13AA7B42B68056" +
            "128322B21F2B6976609B62B6BC4CF2E55FF5AE64E9B68C78A3C2DACC916A1BC7" +
            "322DD353B32898675CFB5B298B176D978B1F12313E3D865BC53465A11CCA1068" +
            "70A4B5D50A2C410938240E92B64902BAEA23EB093D9599E9E372E48336730203" +
            "010001A346304430420603551D01043B3039801024859EBF125E76AF3F0D7979" +
            "B4AC7A96A1133011310F300D060355040313064D794E616D658210D5B5BC1C45" +
            "8A558845BFF51CB4DFF31C300906052B0E03021D0500038181009BF6E2CF830E" +
            "D485B86D6B9E8DFFDCD65EFC7EC145CB9348923710666791FCFA3AB59D689FFD" +
            "7234B7872611C5C23E5E0714531ABADB5DE492D2C736E1C929E648A65CC9EB63" +
            "CD84E57B5909DD5DDF5DBBBA4A6498B9CA225B6E368B94913BFC24DE6B2BD9A2" +
            "6B192B957304B89531E902FFC91B54B237BB228BE8AFCDA264763082020C3082" +
            "0179A00302010202105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03" +
            "021D0500301E311C301A060355040313135253414B65795472616E7366657243" +
            "61706931301E170D3135303431353037303030305A170D323530343135303730" +
            "3030305A301E311C301A060355040313135253414B65795472616E7366657243" +
            "6170693130819F300D06092A864886F70D010101050003818D00308189028181" +
            "00AA272700586C0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB66" +
            "71BA9596C5C63BA2B1AF5C318D9CA39E7400D10C238AC72630579211B86570D1" +
            "A1D44EC86AA8F6C9D2B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD" +
            "910B37DA4093EF76C13B337C1AFAB7D1D07E317B41A336BAA4111299F9942440" +
            "8D0203010001A3533051304F0603551D0104483046801015432DB116B35D07E4" +
            "BA89EDB2469D7AA120301E311C301A060355040313135253414B65795472616E" +
            "73666572436170693182105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B" +
            "0E03021D05000381810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3C" +
            "CF23369FA533C8DC1938952C5931662D9ECD8B1E7B81749E48468167E2FCE3D0" +
            "19FA70D54646975B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D119" +
            "57D653B5C78E5291E4401045576F6D4EDA81BEF3C369AF56121E49A083C8D1AD" +
            "B09F291822E99A42964631820307308203030201013032301E311C301A060355" +
            "040313135253414B65795472616E73666572436170693102105D2FFFF863BABC" +
            "9B4D3C80AB178A4CCA300906052B0E03021A0500300D06092A864886F70D0101" +
            "0105000481805A1717621D450130B3463662160EEC06F7AE77E017DD95F294E9" +
            "7A0BDD433FE6B2CCB34FAAC33AEA50BFD7D9E78DC7174836284619F744278AE7" +
            "7B8495091E096EEF682D9CA95F6E81C7DDCEDDA6A12316B453C894B5000701EB" +
            "09DF57A53B733A4E80DA27FA710870BD88C86E2FDB9DCA14D18BEB2F0C87E963" +
            "2ABF02BE2FE3A182022B3082010806092A864886F70D0109063181FA3081F702" +
            "010380146B4A6B92FDED07EE0119F3674A96D1A70D2A588D300906052B0E0302" +
            "1A0500A03F301806092A864886F70D010903310B06092A864886F70D01070130" +
            "2306092A864886F70D010904311604148C054D6DF2B08E69A86D8DB23C1A5091" +
            "23F9DBA4300D06092A864886F70D0101010500048180962518DEF789B0886C7E" +
            "6295754ECDBDC4CB9D153ECE5EBBE7A82142B92C30DDBBDFC22B5B954F5D844C" +
            "BAEDCA9C4A068B24830E2A96141A5D0320B69EA5DFCFEA441E162D04506F8FFA" +
            "79D7312524F111A9B9B0184007139F94E46C816E0E33F010AEB949F5D884DC89" +
            "87765002F7A643F34B7654E3B2FD5FB34A420279B1EA3082011B06092A864886" +
            "F70D0109063182010C3082010802010130253011310F300D060355040313064D" +
            "794E616D650210D5B5BC1C458A558845BFF51CB4DFF31C300906052B0E03021A" +
            "0500A03F301806092A864886F70D010903310B06092A864886F70D0107013023" +
            "06092A864886F70D010904311604148C054D6DF2B08E69A86D8DB23C1A509123" +
            "F9DBA4300D06092A864886F70D01010105000481801AA282DBED4D862D7CEA30" +
            "F803E790BDB0C97EE852778CEEDDCD94BB9304A1552E60A8D36052AC8C2D2875" +
            "5F3B2F473824100AB3A6ABD4C15ABD77E0FFE13D0DF253BCD99C718FA673B6CB" +
            "0CBBC68CE5A4AC671298C0A07C7223522E0E7FFF15CEDBAB55AAA99588517674" +
            "671691065EB083FB729D1E9C04B2BF99A9953DAA5E").HexToByteArray();

        public static readonly byte[] RsaPkcs1CounterSignedWithNoSignature = (
            "308203E106092A864886F70D010702A08203D2308203CE020101310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08202103082020C30820179A0030201020210" +
            "5D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500301E311C30" +
            "1A060355040313135253414B65795472616E736665724361706931301E170D31" +
            "35303431353037303030305A170D3235303431353037303030305A301E311C30" +
            "1A060355040313135253414B65795472616E73666572436170693130819F300D" +
            "06092A864886F70D010101050003818D0030818902818100AA272700586C0CC4" +
            "1B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA9596C5C63BA2B1" +
            "AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44EC86AA8F6C9D2" +
            "B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37DA4093EF76C1" +
            "3B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203010001A35330" +
            "51304F0603551D0104483046801015432DB116B35D07E4BA89EDB2469D7AA120" +
            "301E311C301A060355040313135253414B65795472616E736665724361706931" +
            "82105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500038181" +
            "0081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF23369FA533C8DC19" +
            "38952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70D54646975B6D" +
            "C2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653B5C78E5291E4" +
            "401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F291822E99A4296" +
            "46318201803082017C0201013032301E311C301A060355040313135253414B65" +
            "795472616E73666572436170693102105D2FFFF863BABC9B4D3C80AB178A4CCA" +
            "300906052B0E03021A0500300D06092A864886F70D01010105000481805A1717" +
            "621D450130B3463662160EEC06F7AE77E017DD95F294E97A0BDD433FE6B2CCB3" +
            "4FAAC33AEA50BFD7D9E78DC7174836284619F744278AE77B8495091E096EEF68" +
            "2D9CA95F6E81C7DDCEDDA6A12316B453C894B5000701EB09DF57A53B733A4E80" +
            "DA27FA710870BD88C86E2FDB9DCA14D18BEB2F0C87E9632ABF02BE2FE3A181A5" +
            "3081A206092A864886F70D010906318194308191020101301C30173115301306" +
            "03550403130C44756D6D79205369676E6572020100300906052B0E03021A0500" +
            "A03F301806092A864886F70D010903310B06092A864886F70D01070130230609" +
            "2A864886F70D010904311604148C054D6DF2B08E69A86D8DB23C1A509123F9DB" +
            "A4300C06082B060105050706020500041466124B3D99FE06A19BBD3C83C593AB" +
            "55D875E28B").HexToByteArray();

        public static readonly byte[] UnsortedSignerInfos = (
            "30820B1E06092A864886F70D010702A0820B0F30820B0B020103310B30090605" +
            "2B0E03021A0500301006092A864886F70D010701A003040107A0820540308202" +
            "0C30820179A00302010202105D2FFFF863BABC9B4D3C80AB178A4CCA30090605" +
            "2B0E03021D0500301E311C301A060355040313135253414B65795472616E7366" +
            "65724361706931301E170D3135303431353037303030305A170D323530343135" +
            "3037303030305A301E311C301A060355040313135253414B65795472616E7366" +
            "6572436170693130819F300D06092A864886F70D010101050003818D00308189" +
            "02818100AA272700586C0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75" +
            "B6EB6671BA9596C5C63BA2B1AF5C318D9CA39E7400D10C238AC72630579211B8" +
            "6570D1A1D44EC86AA8F6C9D2B4E283EA3535923F398A312A23EAEACD8D34FAAC" +
            "A965CD910B37DA4093EF76C13B337C1AFAB7D1D07E317B41A336BAA4111299F9" +
            "9424408D0203010001A3533051304F0603551D0104483046801015432DB116B3" +
            "5D07E4BA89EDB2469D7AA120301E311C301A060355040313135253414B657954" +
            "72616E73666572436170693182105D2FFFF863BABC9B4D3C80AB178A4CCA3009" +
            "06052B0E03021D05000381810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319" +
            "265F3CCF23369FA533C8DC1938952C5931662D9ECD8B1E7B81749E48468167E2" +
            "FCE3D019FA70D54646975B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A0" +
            "75D11957D653B5C78E5291E4401045576F6D4EDA81BEF3C369AF56121E49A083" +
            "C8D1ADB09F291822E99A4296463082032C30820214A003020102020900E0D8AB" +
            "6819D7306E300D06092A864886F70D01010B0500303831363034060355040313" +
            "2D54776F2074686F7573616E6420666F7274792065696768742062697473206F" +
            "662052534120676F6F646E657373301E170D3137313130333233353131355A17" +
            "0D3138313130333233353131355A3038313630340603550403132D54776F2074" +
            "686F7573616E6420666F7274792065696768742062697473206F662052534120" +
            "676F6F646E65737330820122300D06092A864886F70D01010105000382010F00" +
            "3082010A028201010096C114A5898D09133EF859F89C1D848BA8CB5258793E05" +
            "B92D499C55EEFACE274BBBC26803FB813B9C11C6898153CC1745DED2C4D2672F" +
            "807F0B2D957BC4B65EBC9DDE26E2EA7B2A6FE9A7C4D8BD1EF6032B8F0BB6AA33" +
            "C8B57248B3D5E3901D8A38A283D7E25FF8E6F522381EE5484234CFF7B30C1746" +
            "35418FA89E14C468AD89DCFCBBB535E5AF53510F9EA7F9DA8C1B53375B6DAB95" +
            "A291439A5648726EE1012E41388E100691642CF6917F5569D8351F2782F435A5" +
            "79014E8448EEA0C4AECAFF2F476799D88457E2C8BCB56E5E128782B4FE26AFF0" +
            "720D91D52CCAFE344255808F5271D09F784F787E8323182080915BE0AE15A71D" +
            "66476D0F264DD084F30203010001A3393037301D0603551D0E04160414745B5F" +
            "12EF962E84B897E246D399A2BADEA9C5AC30090603551D1304023000300B0603" +
            "551D0F040403020780300D06092A864886F70D01010B0500038201010087A15D" +
            "F37FBD6E9DED7A8FFF25E60B731F635469BA01DD14BC03B2A24D99EFD8B894E9" +
            "493D63EC88C496CB04B33DF25222544F23D43F4023612C4D97B719C1F9431E4D" +
            "B7A580CDF66A3E5F0DAF89A267DD187ABFFB08361B1F79232376AA5FC5AD384C" +
            "C2F98FE36C1CEA0B943E1E3961190648889C8ABE8397A5A338843CBFB1D8B212" +
            "BE46685ACE7B80475CC7C97FC0377936ABD5F664E9C09C463897726650711A11" +
            "10FA9866BC1C278D95E5636AB96FAE95CCD67FD572A8C727E2C03E7B24245731" +
            "8BEC1BE52CA5BD9454A0A41140AE96ED1C56D220D1FD5DD3B1B4FB2AA0E04FC9" +
            "4F7E3C7D476F298962245563953AD7225EDCEAC8B8509E49292E62D8BF318205" +
            "A1308202FB0201038014745B5F12EF962E84B897E246D399A2BADEA9C5AC3009" +
            "06052B0E03021A0500300D06092A864886F70D0101010500048201005E03C5E2" +
            "E736792EFB1C8632C3A864AA6F0E930717FE02C755C0F94DC671244A371926F6" +
            "09878DC8CBFCBA6F83A841B24F48952DA5344F2210BFE9B744E3367B1F8399C8" +
            "96F675923A57E084EBD7DC76A24A1530CD513F0DF6A7703246BF335CC3D09776" +
            "442942150F1C31B9B212AF48850B44B95EB5BD64105F09723EF6AD4711FD81CD" +
            "1FC0418E68EA4428CED9E184126761BF2B25756B6D9BC1A0530E56D38F2A0B78" +
            "3F21D6A5C0703C38F29A2B701B13CAFFCA1DC21C39059E4388E54AEA2519C4E8" +
            "83C7A6BD78200DCB931CA6AB3D18DBBF46A5444C89B6DFE2F48F32C44BA9C030" +
            "F399AC677AA323203137D33CEBFBF1BBF9A506309953B23C4100CA7CA18201C0" +
            "308201BC06092A864886F70D010906318201AD308201A9020101304530383136" +
            "30340603550403132D54776F2074686F7573616E6420666F7274792065696768" +
            "742062697473206F662052534120676F6F646E657373020900E0D8AB6819D730" +
            "6E300906052B0E03021A0500A03F301806092A864886F70D010903310B06092A" +
            "864886F70D010701302306092A864886F70D0109043116041481BF56A6550A60" +
            "A649B0D97971C49897635953D0300D06092A864886F70D010101050004820100" +
            "6E41B7585FEB419005362FEAAAAFB2059E98F8905221A7564F7B0B5510CB221D" +
            "F3DD914A4CD441EAC1C6746A6EC4FC8399C12A61C6B0F50DDA090F564F3D65B2" +
            "6D4BDBC1CE3D39CF47CF33B0D269D15A9FAF2169C60887C3E2CC9828B5E16D45" +
            "DC27A94BAF8D6650EE63D2DBB7DA319B3F61DD18E28AF6FE6DF2CC15C2910BD6" +
            "0B7E038F2C6E8BAEC35CBBBF9484D4C76ECE041DF534B8713B6537854EFE6D58" +
            "41768CCBB9A3B729FDDAE07780CB143A3EE5972DCDDF60A38C65CD3FFF35D1B6" +
            "B76227C1B53831773DA441603F4FB5764D33AADE102F9B85D2CDAEC0E3D6C6E8" +
            "C24C434BFAA3E12E02202142784ED0EB2D9CDCC276D21474747DCD3E4F4D54FC" +
            "3081D40201013032301E311C301A060355040313135253414B65795472616E73" +
            "666572436170693102105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E" +
            "03021A0500300D06092A864886F70D01010105000481805EB33C6A9ED5B62240" +
            "90C431E79F51D70B4F2A7D31ED4ED8C3465F6E01281C3FFA44116238B2D168D8" +
            "9154136DDB8B4EB31EA685FB719B7384510F5EF077A10DE6A5CA86F4F6D28B58" +
            "79AFD6CFF0BDA005C2D7CFF53620D28988CBAA44F18AA2D50229FA930B0A7262" +
            "D780DFDEC0334A97DF872F1D95087DC11A881568AF5B88308201C70201013045" +
            "3038313630340603550403132D54776F2074686F7573616E6420666F72747920" +
            "65696768742062697473206F662052534120676F6F646E657373020900E0D8AB" +
            "6819D7306E300906052B0E03021A0500A05D301806092A864886F70D01090331" +
            "0B06092A864886F70D010701301C06092A864886F70D010905310F170D313731" +
            "3130393136303934315A302306092A864886F70D010904311604145D1BE7E9DD" +
            "A1EE8896BE5B7E34A85EE16452A7B4300D06092A864886F70D01010105000482" +
            "01000BB9410F23CFD9C1FCB16179612DB871224F5B88A8E2C012DCDBB3699780" +
            "A3311FD330FFDD6DF1434C52DADD6E07D81FEF145B806E71AF471223914B98CD" +
            "588CCCDFB50ABE3D991B11D62BD83DE158A9001BAED3549BC49B8C204D25C17B" +
            "D042756B026692959E321ACC1AFE6BF52C9356FD49936116D2B3D1F6569F8A8B" +
            "F0FBB2E403AD5788681F3AD131E57390ACB9B8C2EA0BE717F22EFE577EFB1063" +
            "6AC465469191B7E4B3F03CF8DC6C310A20D2B0891BC27350C7231BC2EAABF129" +
            "83755B4C0EDF8A0EE99A615D4E8B381C67A7CDB1405D98C2A6285FEDCED5A65F" +
            "C45C31CD33E3CEB96223DB45E9156B9BD7C8E442C40ED1BB6866C03548616061" +
            "3DAF").HexToByteArray();

        public static byte[] OneDsa1024 = (
            "3082044206092A864886F70D010702A08204333082042F020103310B30090605" +
            "2B0E03021A0500302406092A864886F70D010701A01704154D6963726F736F66" +
            "7420436F72706F726174696F6EA08203913082038D3082034AA0030201020209" +
            "00AB740A714AA83C92300B060960864801650304030230818D310B3009060355" +
            "040613025553311330110603550408130A57617368696E67746F6E3110300E06" +
            "0355040713075265646D6F6E64311E301C060355040A13154D6963726F736F66" +
            "7420436F72706F726174696F6E3120301E060355040B13172E4E455420467261" +
            "6D65776F726B2028436F7265465829311530130603550403130C313032342D62" +
            "697420445341301E170D3135313132353134343030335A170D31353132323531" +
            "34343030335A30818D310B300906035504061302555331133011060355040813" +
            "0A57617368696E67746F6E3110300E060355040713075265646D6F6E64311E30" +
            "1C060355040A13154D6963726F736F667420436F72706F726174696F6E312030" +
            "1E060355040B13172E4E4554204672616D65776F726B2028436F726546582931" +
            "1530130603550403130C313032342D62697420445341308201B73082012C0607" +
            "2A8648CE3804013082011F02818100AEE3309FC7C9DB750D4C3797D333B3B9B2" +
            "34B462868DB6FFBDED790B7FC8DDD574C2BD6F5E749622507AB2C09DF5EAAD84" +
            "859FC0706A70BB8C9C8BE22B4890EF2325280E3A7F9A3CE341DBABEF6058D063" +
            "EA6783478FF8B3B7A45E0CA3F7BAC9995DCFDDD56DF168E91349130F719A4E71" +
            "7351FAAD1A77EAC043611DC5CC5A7F021500D23428A76743EA3B49C62EF0AA17" +
            "314A85415F0902818100853F830BDAA738465300CFEE02418E6B07965658EAFD" +
            "A7E338A2EB1531C0E0CA5EF1A12D9DDC7B550A5A205D1FF87F69500A4E4AF575" +
            "9F3F6E7F0C48C55396B738164D9E35FB506BD50E090F6A497C70E7E868C61BD4" +
            "477C1D62922B3DBB40B688DE7C175447E2E826901A109FAD624F1481B276BF63" +
            "A665D99C87CEE9FD06330381840002818025B8E7078E149BAC35266762362002" +
            "9F5E4A5D4126E336D56F1189F9FF71EA671B844EBD351514F27B69685DDF716B" +
            "32F102D60EA520D56F544D19B2F08F5D9BDDA3CBA3A73287E21E559E6A075861" +
            "94AFAC4F6E721EDCE49DE0029627626D7BD30EEB337311DB4FF62D7608997B6C" +
            "C32E9C42859820CA7EF399590D5A388C48A330302E302C0603551D1104253023" +
            "87047F00000187100000000000000000000000000000000182096C6F63616C68" +
            "6F7374300B0609608648016503040302033000302D021500B9316CC7E05C9F79" +
            "197E0B41F6FD4E3FCEB72A8A0214075505CCAECB18B7EF4C00F9C069FA3BC780" +
            "14DE31623060020103801428A2CB1D204C2656A79C931EFAE351AB548248D030" +
            "0906052B0E03021A0500300906072A8648CE380403042F302D021476DCB780CE" +
            "D5B308A3630726A85DB97FBC50DFD1021500CDF2649B50500BB7428B9DCA6BEF" +
            "2C7E7EF1B79C").HexToByteArray();

        internal static readonly byte[] SignedCmsOverEnvelopedCms_IssuerSerial_NetFx = (
            "3082047C06092A864886F70D010702A082046D30820469020101310B30090605" +
            "2B0E03021A05003082012406092A864886F70D010703A0820115308201110609" +
            "2A864886F70D010703A08201023081FF0201003181CC3081C90201003032301E" +
            "311C301A060355040313135253414B65795472616E7366657243617069310210" +
            "5D2FFFF863BABC9B4D3C80AB178A4CCA300D06092A864886F70D010101050004" +
            "81800BB53BF3BD028A6B54703899B241CB358CACBF9018A4497A733C27EA223E" +
            "05BD31099EB80AE04ADBB23A5E397C181A14476668402EFE3BCA08BCA615C743" +
            "41FA06D56671AA940BF09B6B7B4C6905AD2927DE94960ED03DF141360589979F" +
            "9944DB48B91AA1B139EB652D6A1BAC48DF33AF14006CD9DB4C09E7DA270733D0" +
            "DF90302B06092A864886F70D010701301406082A864886F70D03070408E4972B" +
            "4188B1B4FE80084CBF0A9D37B094EBA08202103082020C30820179A003020102" +
            "02105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D0500301E31" +
            "1C301A060355040313135253414B65795472616E736665724361706931301E17" +
            "0D3135303431353037303030305A170D3235303431353037303030305A301E31" +
            "1C301A060355040313135253414B65795472616E73666572436170693130819F" +
            "300D06092A864886F70D010101050003818D0030818902818100AA272700586C" +
            "0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA9596C5C63B" +
            "A2B1AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44EC86AA8F6" +
            "C9D2B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37DA4093EF" +
            "76C13B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203010001A3" +
            "533051304F0603551D0104483046801015432DB116B35D07E4BA89EDB2469D7A" +
            "A120301E311C301A060355040313135253414B65795472616E73666572436170" +
            "693182105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D050003" +
            "81810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF23369FA533C8" +
            "DC1938952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70D5464697" +
            "5B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653B5C78E52" +
            "91E4401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F291822E99A" +
            "42964631820119308201150201013032301E311C301A06035504031313525341" +
            "4B65795472616E73666572436170693102105D2FFFF863BABC9B4D3C80AB178A" +
            "4CCA300906052B0E03021A0500A03F301806092A864886F70D010903310B0609" +
            "2A864886F70D010703302306092A864886F70D01090431160414FE46C861E86B" +
            "719D0F665AFAE48165B56CDFBFD4300D06092A864886F70D0101010500048180" +
            "32CEE36532673C2734C908A48B6E017FD695BE69FAC21028B6627466B72688D8" +
            "60FC65F2F18E5C19FED2301351F247DF90217087C5F88D76CA052287E6A2F47F" +
            "7DA5AC226B4FC202AB0B5B73A24B5C138247F54466621288F2DA941320C4CE89" +
            "A503ED3E6F63112798A841E55344BEE84E1366E4CF3788C9788C5E86D1879029").HexToByteArray();

        internal static readonly byte[] SignedCmsOverEnvelopedCms_SKID_NetFx = (
            "3082046006092A864886F70D010702A08204513082044D020103310B30090605" +
            "2B0E03021A05003082012806092A864886F70D010703A0820119048201153082" +
            "011106092A864886F70D010703A08201023081FF0201003181CC3081C9020100" +
            "3032301E311C301A060355040313135253414B65795472616E73666572436170" +
            "693102105D2FFFF863BABC9B4D3C80AB178A4CCA300D06092A864886F70D0101" +
            "0105000481803ECF128C059F49199D3344979BD0EBAC2A5443D4F27775B8CFAC" +
            "7B1F28AFDDAD86097FF34DFB3ED2D514C325B78074D6D17CA14952EA954E860B" +
            "D5980F2C629C70AE402D3E9E867246E532E345712DFA33C37EF141E2EBFD10F7" +
            "249CFD193B313825CB7B297FB204DA755F02384659F51D97AB31F867C7E973C6" +
            "28B9F6E43018302B06092A864886F70D010701301406082A864886F70D030704" +
            "089FC5129D8AB0CDDE80086D7E35774EFA334AA08202103082020C30820179A0" +
            "0302010202105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E03021D05" +
            "00301E311C301A060355040313135253414B65795472616E7366657243617069" +
            "31301E170D3135303431353037303030305A170D323530343135303730303030" +
            "5A301E311C301A060355040313135253414B65795472616E7366657243617069" +
            "3130819F300D06092A864886F70D010101050003818D0030818902818100AA27" +
            "2700586C0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671BA95" +
            "96C5C63BA2B1AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1D44E" +
            "C86AA8F6C9D2B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD910B37" +
            "DA4093EF76C13B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D0203" +
            "010001A3533051304F0603551D0104483046801015432DB116B35D07E4BA89ED" +
            "B2469D7AA120301E311C301A060355040313135253414B65795472616E736665" +
            "72436170693182105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E0302" +
            "1D05000381810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF2336" +
            "9FA533C8DC1938952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019FA70" +
            "D54646975B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957D653" +
            "B5C78E5291E4401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB09F29" +
            "1822E99A4296463181FA3081F702010380146B4A6B92FDED07EE0119F3674A96" +
            "D1A70D2A588D300906052B0E03021A0500A03F301806092A864886F70D010903" +
            "310B06092A864886F70D010703302306092A864886F70D0109043116041435DE" +
            "A4AE3B383A023271BA27D2D50EC021D40800300D06092A864886F70D01010105" +
            "00048180386A2EB06AB0ED0111EB37214480CD782243C66105948AD8EAB3236A" +
            "7ECF135F22B6558F3C601140F6BBDF313F7DB98B3E6277ED5C2407D57323348D" +
            "A97F6A9653C7C219EE1B0E3F85A970FA6CFC00B53E72484F732916E6067E2F0D" +
            "4D31EFF51CECD46F3EF245FEF8729C4E1F16C0A3054054477D6C787FC7C94D79" +
            "A24AC54B").HexToByteArray();

        internal static readonly byte[] SignedCmsOverEnvelopedCms_IssuerSerial_CoreFx = (
            "3082048E06092A864886F70D010702A082047F3082047B020103310D300B0609" +
            "6086480165030402013082012806092A864886F70D010703A082011904820115" +
            "3082011106092A864886F70D010703A08201023081FF0201003181CC3081C902" +
            "01003032301E311C301A060355040313135253414B65795472616E7366657243" +
            "6170693102105D2FFFF863BABC9B4D3C80AB178A4CCA300D06092A864886F70D" +
            "01010105000481801B7806566B26A92076D5C9F5A06FBC9AB1D53BD63D3B7F97" +
            "569B683219C4BA0B285F2F3EF533387EDD7E6BE38DFDD1F33EBA8E5001238BD0" +
            "E75B9A5C5E2504FD78954B372A2E8B183F4CBD2D239CB72D129E112D0476D9A9" +
            "A00AF0EC700776F4719BC4838DBAC7F06C671F67B977ABDF449B42C98D28035A" +
            "194CE2B786E8C8A2302B06092A864886F70D010701301406082A864886F70D03" +
            "070408B4B41A525B6E8F628008767424A015173966A08202103082020C308201" +
            "79A00302010202105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E0302" +
            "1D0500301E311C301A060355040313135253414B65795472616E736665724361" +
            "706931301E170D3135303431353037303030305A170D32353034313530373030" +
            "30305A301E311C301A060355040313135253414B65795472616E736665724361" +
            "70693130819F300D06092A864886F70D010101050003818D0030818902818100" +
            "AA272700586C0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671" +
            "BA9596C5C63BA2B1AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1" +
            "D44EC86AA8F6C9D2B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD91" +
            "0B37DA4093EF76C13B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D" +
            "0203010001A3533051304F0603551D0104483046801015432DB116B35D07E4BA" +
            "89EDB2469D7AA120301E311C301A060355040313135253414B65795472616E73" +
            "666572436170693182105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E" +
            "03021D05000381810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF" +
            "23369FA533C8DC1938952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019" +
            "FA70D54646975B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957" +
            "D653B5C78E5291E4401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB0" +
            "9F291822E99A42964631820125308201210201013032301E311C301A06035504" +
            "0313135253414B65795472616E73666572436170693102105D2FFFF863BABC9B" +
            "4D3C80AB178A4CCA300B0609608648016503040201A04B301806092A864886F7" +
            "0D010903310B06092A864886F70D010703302F06092A864886F70D0109043122" +
            "042018BEF3F24109B4BCD5BF3D5372EA7A0D16AF6DF46DE9BE5C2373DF065381" +
            "5E13300B06092A864886F70D01010104818016A02798B3CEC42BE258C85A4BED" +
            "06099339C9E716B8C72A3330923BE4B6A0538A5DCE031CD710589E8281E24074" +
            "F26AB6B86CEACF78449B82FF1512F511B5A97ABA4403029E2BA1D837D3F9D230" +
            "45E0EB3CE59E3AF7E52B814EFCBBCFD7A442327C5C408D166D4302AEFF807ECB" +
            "D107C811DC66EC35FE167408B58FB03B7F84").HexToByteArray();

        internal static readonly byte[] SignedCmsOverEnvelopedCms_SKID_CoreFx = (
            "3082047006092A864886F70D010702A08204613082045D020103310D300B0609" +
            "6086480165030402013082012806092A864886F70D010703A082011904820115" +
            "3082011106092A864886F70D010703A08201023081FF0201003181CC3081C902" +
            "01003032301E311C301A060355040313135253414B65795472616E7366657243" +
            "6170693102105D2FFFF863BABC9B4D3C80AB178A4CCA300D06092A864886F70D" +
            "0101010500048180724D9D5E0D2110B8147589120524B1D1E7019A3F436AD459" +
            "3DF555413423AE28FCBA01548B20FDCA21901ECF6B54331542CECD4326C7E292" +
            "54AA563D7F38C2287C146B648E6779FA3843FB0F11A3726265266DF87BAAF04B" +
            "AA1DD4825B9FFFEBD1DC47414EA4978580A03484B9159E57045018DAA3054704" +
            "84046F89465169A0302B06092A864886F70D010701301406082A864886F70D03" +
            "0704087E74D74C2652F5198008930CBA811F9E9E15A08202103082020C308201" +
            "79A00302010202105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E0302" +
            "1D0500301E311C301A060355040313135253414B65795472616E736665724361" +
            "706931301E170D3135303431353037303030305A170D32353034313530373030" +
            "30305A301E311C301A060355040313135253414B65795472616E736665724361" +
            "70693130819F300D06092A864886F70D010101050003818D0030818902818100" +
            "AA272700586C0CC41B05C65C7D846F5A2BC27B03E301C37D9BFF6D75B6EB6671" +
            "BA9596C5C63BA2B1AF5C318D9CA39E7400D10C238AC72630579211B86570D1A1" +
            "D44EC86AA8F6C9D2B4E283EA3535923F398A312A23EAEACD8D34FAACA965CD91" +
            "0B37DA4093EF76C13B337C1AFAB7D1D07E317B41A336BAA4111299F99424408D" +
            "0203010001A3533051304F0603551D0104483046801015432DB116B35D07E4BA" +
            "89EDB2469D7AA120301E311C301A060355040313135253414B65795472616E73" +
            "666572436170693182105D2FFFF863BABC9B4D3C80AB178A4CCA300906052B0E" +
            "03021D05000381810081E5535D8ECEEF265ACBC82F6C5F8BC9D84319265F3CCF" +
            "23369FA533C8DC1938952C5931662D9ECD8B1E7B81749E48468167E2FCE3D019" +
            "FA70D54646975B6DC2A3BA72D5A5274C1866DA6D7A5DF47938E034A075D11957" +
            "D653B5C78E5291E4401045576F6D4EDA81BEF3C369AF56121E49A083C8D1ADB0" +
            "9F291822E99A429646318201073082010302010380146B4A6B92FDED07EE0119" +
            "F3674A96D1A70D2A588D300B0609608648016503040201A04B301806092A8648" +
            "86F70D010903310B06092A864886F70D010703302F06092A864886F70D010904" +
            "31220420873B6A3B7CE192922129761C3EDD8D68C4A6B0369F3BF5B3D30B0A9E" +
            "2336A8F4300B06092A864886F70D0101010481807D31B3260AE00DE3992DDD1E" +
            "B01FDECA28053F2B87AA723CCD27B92896E3199F7C4B3B4A391C181899E5CBD1" +
            "4A4BCDDFF6DC6CD10CA118DAA62E32589F066D1669D2948E51B5363B7BEE2BA9" +
            "351CDE1791D118E552F0C8A4FB58EC7C34F5BAB2D562B415C4B3F673179B8410" +
            "86A9B0F03ED56DBD4FA9CBB775307C9BB3045F72").HexToByteArray();

        // This document signature was generated without sorting the attributes,
        // it ensures a compatibility fallback in CheckSignature.
        internal static readonly byte[] DigiCertTimeStampToken = (
            "30820EC406092A864886F70D010702A0820EB530820EB1020103310F300D0609" +
            "608648016503040201050030818A060B2A864886F70D0109100104A07B047930" +
            "7702010106096086480186FD6C07013031300D06096086480165030402010500" +
            "04205A3B99F86CE06F5639C3ED7D0E16AE44A5CD58ABA0F6853A2AF389F91B6C" +
            "7D7F02104E640CDE5C761BFF32FA0ADEA58C5F56180F32303138303731373136" +
            "313732315A0211457A3006B9483730D09772076C4B74BDF8A0820BBB30820682" +
            "3082056AA003020102021009C0FC46C8044213B5598BAF284F4E41300D06092A" +
            "864886F70D01010B05003072310B300906035504061302555331153013060355" +
            "040A130C446967694365727420496E6331193017060355040B13107777772E64" +
            "696769636572742E636F6D3131302F0603550403132844696769436572742053" +
            "48413220417373757265642049442054696D657374616D70696E67204341301E" +
            "170D3137303130343030303030305A170D3238303131383030303030305A304C" +
            "310B30090603550406130255533111300F060355040A13084469676943657274" +
            "312A302806035504031321446967694365727420534841322054696D65737461" +
            "6D7020526573706F6E64657230820122300D06092A864886F70D010101050003" +
            "82010F003082010A02820101009E95986A343B731BA87EFCC7BE296989C76826" +
            "465F3D8D62738781A3A19CF0B75B24375A92D4F459D77689E4DCD527F0D566BC" +
            "0AEEB42B3167AC58C54A91592B451E0901D664B359EE8D664DFB235ECC100D0B" +
            "8A67EF52AEA00890C252F7F5A8B56E9B2C7B9DE7B53EFB78CD325018BF40B54C" +
            "8CBB57F4A04F11456C4242B9E5AFD6DFF4A77C0A68960FD25F2957CEFB1D32FF" +
            "F411A11322FB12CBEFD753D2EB97CBA2AC1B1D9D58215182C2C2DEEA2B3F2C22" +
            "84D043EC3B3B3F47C4F656DC453798B46B74B559AF785769C80F090278DDD853" +
            "C199DB60C49DEAAEAFE07E864A5CA95861A85E748A012868724EA7869DB50252" +
            "87706648D38EEF8124CCDCD8650203010001A382033830820334300E0603551D" +
            "0F0101FF040403020780300C0603551D130101FF0402300030160603551D2501" +
            "01FF040C300A06082B06010505070308308201BF0603551D20048201B6308201" +
            "B2308201A106096086480186FD6C070130820192302806082B06010505070201" +
            "161C68747470733A2F2F7777772E64696769636572742E636F6D2F4350533082" +
            "016406082B06010505070202308201561E8201520041006E0079002000750073" +
            "00650020006F0066002000740068006900730020004300650072007400690066" +
            "0069006300610074006500200063006F006E0073007400690074007500740065" +
            "007300200061006300630065007000740061006E006300650020006F00660020" +
            "007400680065002000440069006700690043006500720074002000430050002F" +
            "00430050005300200061006E00640020007400680065002000520065006C0079" +
            "0069006E0067002000500061007200740079002000410067007200650065006D" +
            "0065006E00740020007700680069006300680020006C0069006D006900740020" +
            "006C0069006100620069006C00690074007900200061006E0064002000610072" +
            "006500200069006E0063006F00720070006F0072006100740065006400200068" +
            "0065007200650069006E0020006200790020007200650066006500720065006E" +
            "00630065002E300B06096086480186FD6C0315301F0603551D23041830168014" +
            "F4B6E1201DFE29AED2E461A5B2A225B2C817356E301D0603551D0E04160414E1" +
            "A7324AEE0121287D54D5F207926EB4070F3D8730710603551D1F046A30683032" +
            "A030A02E862C687474703A2F2F63726C332E64696769636572742E636F6D2F73" +
            "6861322D617373757265642D74732E63726C3032A030A02E862C687474703A2F" +
            "2F63726C342E64696769636572742E636F6D2F736861322D617373757265642D" +
            "74732E63726C30818506082B0601050507010104793077302406082B06010505" +
            "0730018618687474703A2F2F6F6373702E64696769636572742E636F6D304F06" +
            "082B060105050730028643687474703A2F2F636163657274732E646967696365" +
            "72742E636F6D2F44696769436572745348413241737375726564494454696D65" +
            "7374616D70696E6743412E637274300D06092A864886F70D01010B0500038201" +
            "01001EF0418232AEEDF1B43513DC50C2D597AE22229D0E0EAF33D34CFD7CBF6F" +
            "0111A79465225CC622A1C889526B9A8C735CD95E3F32DE16604C8B36FD31990A" +
            "BDC184B78D1DEF8926130556F347CD475BAD84B238AF6A23B545E31E88324680" +
            "D2B7A69922FDC178CFF58BD80C8C0509EE44E680D56D70CC9F531E27DD2A48DE" +
            "DA9365AD6E65A399A7C2400E73CC584F8F4528E5BC9C88E628CE605D2D255D8B" +
            "732EA50D5B51DA9A4EFF50058928DAF278BBD258788D44A7AC3A009178698964" +
            "04D35D96DF2ABFF9A54C2C93FFE68ADD82ACF1D2B3A2869AC15589566A473FFA" +
            "D6339543358905785A3A69DA22B80443D36F6835367A143E45E99864860F130C" +
            "264A3082053130820419A00302010202100AA125D6D6321B7E41E405DA3697C2" +
            "15300D06092A864886F70D01010B05003065310B300906035504061302555331" +
            "153013060355040A130C446967694365727420496E6331193017060355040B13" +
            "107777772E64696769636572742E636F6D312430220603550403131B44696769" +
            "43657274204173737572656420494420526F6F74204341301E170D3136303130" +
            "373132303030305A170D3331303130373132303030305A3072310B3009060355" +
            "04061302555331153013060355040A130C446967694365727420496E63311930" +
            "17060355040B13107777772E64696769636572742E636F6D3131302F06035504" +
            "0313284469676943657274205348413220417373757265642049442054696D65" +
            "7374616D70696E6720434130820122300D06092A864886F70D01010105000382" +
            "010F003082010A0282010100BDD032EE4BCD8F7FDDA9BA8299C539542857B623" +
            "4AC40E07453351107DD0F97D4D687EE7B6A0F48DB388E497BF63219098BF13BC" +
            "57D3C3E17E08D66A140038F72E1E3BEECCA6F63259FE5F653FE09BEBE3464706" +
            "1A557E0B277EC0A2F5A0E0DE223F0EFF7E95FBF3A3BA223E18AC11E4F099036D" +
            "3B857C09D3EE5DC89A0B54E3A809716BE0CF22100F75CF71724E0AADDF403A5C" +
            "B751E1A17914C64D2423305DBCEC3C606AAC2F07CCFDF0EA47D988505EFD666E" +
            "56612729898451E682E74650FD942A2CA7E4753EBA980F847F9F3114D6ADD5F2" +
            "64CB7B1E05D084197217F11706EF3DCDD64DEF0642FDA2532A4F851DC41D3CAF" +
            "CFDAAC10F5DDACACE956FF930203010001A38201CE308201CA301D0603551D0E" +
            "04160414F4B6E1201DFE29AED2E461A5B2A225B2C817356E301F0603551D2304" +
            "183016801445EBA2AFF492CB82312D518BA7A7219DF36DC80F30120603551D13" +
            "0101FF040830060101FF020100300E0603551D0F0101FF040403020186301306" +
            "03551D25040C300A06082B06010505070308307906082B06010505070101046D" +
            "306B302406082B060105050730018618687474703A2F2F6F6373702E64696769" +
            "636572742E636F6D304306082B060105050730028637687474703A2F2F636163" +
            "657274732E64696769636572742E636F6D2F4469676943657274417373757265" +
            "644944526F6F7443412E6372743081810603551D1F047A3078303AA038A03686" +
            "34687474703A2F2F63726C342E64696769636572742E636F6D2F446967694365" +
            "7274417373757265644944526F6F7443412E63726C303AA038A0368634687474" +
            "703A2F2F63726C332E64696769636572742E636F6D2F44696769436572744173" +
            "73757265644944526F6F7443412E63726C30500603551D20044930473038060A" +
            "6086480186FD6C000204302A302806082B06010505070201161C68747470733A" +
            "2F2F7777772E64696769636572742E636F6D2F435053300B06096086480186FD" +
            "6C0701300D06092A864886F70D01010B05000382010100719512E951875669CD" +
            "EFDDDA7CAA637AB378CF06374084EF4B84BFCACF0302FDC5A7C30E20422CAF77" +
            "F32B1F0C215A2AB705341D6AAE99F827A266BF09AA60DF76A43A930FF8B2D1D8" +
            "7C1962E85E82251EC4BA1C7B2C21E2D65B2C1435430468B2DB7502E072C798D6" +
            "3C64E51F4810185F8938614D62462487638C91522CAF2989E5781FD60B14A580" +
            "D7124770B375D59385937EB69267FB536189A8F56B96C0F458690D7CC801B1B9" +
            "2875B7996385228C61CA79947E59FC8C0FE36FB50126B66CA5EE875121E45860" +
            "9BBA0C2D2B6DA2C47EBBC4252B4702087C49AE13B6E17C424228C61856CF4134" +
            "B6665DB6747BF55633222F2236B24BA24A95D8F5A68E523182024D3082024902" +
            "01013081863072310B300906035504061302555331153013060355040A130C44" +
            "6967694365727420496E6331193017060355040B13107777772E646967696365" +
            "72742E636F6D3131302F06035504031328446967694365727420534841322041" +
            "7373757265642049442054696D657374616D70696E67204341021009C0FC46C8" +
            "044213B5598BAF284F4E41300D06096086480165030402010500A08198301A06" +
            "092A864886F70D010903310D060B2A864886F70D0109100104301C06092A8648" +
            "86F70D010905310F170D3138303731373136313732315A302F06092A864886F7" +
            "0D01090431220420E66606A88254749C2E5575722F93AA67174FDDDB7703B7E6" +
            "FAD6B3FE000F3DE2302B060B2A864886F70D010910020C311C301A3018301604" +
            "14400191475C98891DEBA104AF47091B5EB6D4CBCB300D06092A864886F70D01" +
            "01010500048201005AF349DE87550378C702ED31AE6DD6D50E6B24298DB2DFD6" +
            "1396C6FA3E465FE7323ACD65AE157C06BCB993551F33702C6D1F6951AECDE74A" +
            "969E41A8F0F95188780F990EAF6B129633CDE42FF149501BFAC05C516B6DA9EF" +
            "E492488013928BA801D66C32EFE7EEDFF22DC96DDA4783674EEE8231E7A3AD8A" +
            "98A506DABB68D6337D7FFDBD2F7112AF2FEE718CF6E7E5544DB7B4BDCD8191EB" +
            "C73D568EE4D2A30B8478D676910E3B4EB868010AAF22400198D0593F987C86A9" +
            "101711B9C6AC5C5776923C699E772B07864755C1AC50F387655C4E67DB356207" +
            "76252A2F4605B97BD3C299D1CD79929273BB86E7DF9E113C92802380ED6D4041" +
            "9DA4C01214D4FA24").HexToByteArray();

        internal static readonly byte[] RsaPkcs1Sha256WithRsa = (
            "3082071B06092A864886F70D010702A082070C30820708020101310F300D0609" +
            "6086480165030402010500301606092A864886F70D010701A009040700010203" +
            "040506A08205223082051E30820406A00302010202100D85090F3FACFF0A9008" +
            "A12A9FB00A54300D06092A864886F70D01010B05003072310B30090603550406" +
            "1302555331153013060355040A130C446967694365727420496E633119301706" +
            "0355040B13107777772E64696769636572742E636F6D3131302F060355040313" +
            "2844696769436572742053484132204173737572656420494420436F64652053" +
            "69676E696E67204341301E170D3138303832393030303030305A170D31393039" +
            "30333132303030305A305B310B3009060355040613025553310B300906035504" +
            "0813025641311330110603550407130A416C6578616E64726961311430120603" +
            "55040A130B4B6576696E204A6F6E6573311430120603550403130B4B6576696E" +
            "204A6F6E657330820122300D06092A864886F70D01010105000382010F003082" +
            "010A0282010100F1F4542FF6CA57FBC44986EC816F07D1FD50BFD477C412D299" +
            "1C962D0A22194A4296BCD0751F47CE4932F73871277CE3CDD2C78157599C7A35" +
            "80CC96A11F7031E3A798F4BAA93988F0E4077D30316252B24337DB26914E1F77" +
            "9CB4979544514B0234E5388E936B195B91863B258F0C8951454D3668F0C4D456" +
            "A8497758D21C433626E46F2CFF5A7CC7945F788948998E5F8786E1E990E240BB" +
            "0780CD258F57761AFB5D42AD8E3D703C3126861E83F191ECE9F0B83221F96214" +
            "533B2A47977F43715FE501FBC4A4040839DD3EBCA8B67259A7DD0EA9EFAE2200" +
            "943EFB7D0404B8978B49A445849B5F6898B06269F427F30DBC8DB2FD7963943A" +
            "8C461760E6A4F30203010001A38201C5308201C1301F0603551D230418301680" +
            "145AC4B97B2A0AA3A5EA7103C060F92DF665750E58301D0603551D0E04160414" +
            "33795EB2D84BFAA3F96E5930F64EC6A94C6FD36A300E0603551D0F0101FF0404" +
            "0302078030130603551D25040C300A06082B0601050507030330770603551D1F" +
            "0470306E3035A033A031862F687474703A2F2F63726C332E6469676963657274" +
            "2E636F6D2F736861322D617373757265642D63732D67312E63726C3035A033A0" +
            "31862F687474703A2F2F63726C342E64696769636572742E636F6D2F73686132" +
            "2D617373757265642D63732D67312E63726C304C0603551D2004453043303706" +
            "096086480186FD6C0301302A302806082B06010505070201161C68747470733A" +
            "2F2F7777772E64696769636572742E636F6D2F4350533008060667810C010401" +
            "30818406082B0601050507010104783076302406082B06010505073001861868" +
            "7474703A2F2F6F6373702E64696769636572742E636F6D304E06082B06010505" +
            "0730028642687474703A2F2F636163657274732E64696769636572742E636F6D" +
            "2F446967694365727453484132417373757265644944436F64655369676E696E" +
            "6743412E637274300C0603551D130101FF04023000300D06092A864886F70D01" +
            "010B0500038201010045B9D9868E494BD02635D0E42DDE80B37A865C389CFDD9" +
            "9BFC9B62E2C169A73B5EABF282607439EFF5C61630886DEB63415B53683446A7" +
            "3041686C326BA35FF0029FEF603D7C80FA0177A4DE35013529B01F759FD50414" +
            "79BDBB6B93B18144CB14E431BC144146848EF8ADB0E28952EAD1BB49E8547FFE" +
            "9934817036338B20C4E0B9D7C6A4E5BE3D57157F21904A5C864946313EA6B7D9" +
            "50EE0235B5D2CD01490AD2B2A1AB5F66EC8986D64A1D9D239C131E09E5CA1C02" +
            "A75F2D7EC07E4C858856A6A58AB94DEAC8B3D3A5BBF492EE2463B156E6A0660B" +
            "B452E35922D00456F0DEE0ED15A8BF8FFF31008756B14EEE0AC14BCF19A3CD16" +
            "819DC990F5F45CDE21318201B2308201AE0201013081863072310B3009060355" +
            "04061302555331153013060355040A130C446967694365727420496E63311930" +
            "17060355040B13107777772E64696769636572742E636F6D3131302F06035504" +
            "03132844696769436572742053484132204173737572656420494420436F6465" +
            "205369676E696E6720434102100D85090F3FACFF0A9008A12A9FB00A54300D06" +
            "096086480165030402010500300D06092A864886F70D01010B050004820100E2" +
            "980C5A30EC00729D1CFA826D7A65B43FF6806B5E0ABA23A78E4F1CAA3F6436EF" +
            "00941C6947A9B8F20D0757B5346CF640AA217F7361BEEFF2BC997FB1D3597BF3" +
            "D7457BD4A94062FB03660F9D86710BE2FC99876A848251F4965E1B16192714C8" +
            "F9788C09CCDE83603ADC919297BA496E921B95F3BD9554A873E09912640FCFAA" +
            "D9DD1441D1851E637031D390C038223AE64B048E806462DDBAC98C156BE2EE47" +
            "2B78166BDB1612848B535ADC3F0E7BE52991A17F48AFDCCC1698A236BA338930" +
            "50EBAAC4460DAA35185C16670F597E0E6E0CB0AA83F51AAEF452F3367DD9350A" +
            "8A49A5A8F79DF8E921303AB5D6646A482F0F59D9980310E1AE3EE8D77CB857").HexToByteArray();

        internal static readonly byte[] RsaPkcs1SignedSha1DeclaredSha256WithRsa = (
            "3082071306092A864886F70D010702A082070430820700020101310B30090605" +
            "2B0E03021A0500301606092A864886F70D010701A009040700010203040506A0" +
            "8205223082051E30820406A00302010202100D85090F3FACFF0A9008A12A9FB0" +
            "0A54300D06092A864886F70D01010B05003072310B3009060355040613025553" +
            "31153013060355040A130C446967694365727420496E6331193017060355040B" +
            "13107777772E64696769636572742E636F6D3131302F06035504031328446967" +
            "69436572742053484132204173737572656420494420436F6465205369676E69" +
            "6E67204341301E170D3138303832393030303030305A170D3139303930333132" +
            "303030305A305B310B3009060355040613025553310B30090603550408130256" +
            "41311330110603550407130A416C6578616E6472696131143012060355040A13" +
            "0B4B6576696E204A6F6E6573311430120603550403130B4B6576696E204A6F6E" +
            "657330820122300D06092A864886F70D01010105000382010F003082010A0282" +
            "010100F1F4542FF6CA57FBC44986EC816F07D1FD50BFD477C412D2991C962D0A" +
            "22194A4296BCD0751F47CE4932F73871277CE3CDD2C78157599C7A3580CC96A1" +
            "1F7031E3A798F4BAA93988F0E4077D30316252B24337DB26914E1F779CB49795" +
            "44514B0234E5388E936B195B91863B258F0C8951454D3668F0C4D456A8497758" +
            "D21C433626E46F2CFF5A7CC7945F788948998E5F8786E1E990E240BB0780CD25" +
            "8F57761AFB5D42AD8E3D703C3126861E83F191ECE9F0B83221F96214533B2A47" +
            "977F43715FE501FBC4A4040839DD3EBCA8B67259A7DD0EA9EFAE2200943EFB7D" +
            "0404B8978B49A445849B5F6898B06269F427F30DBC8DB2FD7963943A8C461760" +
            "E6A4F30203010001A38201C5308201C1301F0603551D230418301680145AC4B9" +
            "7B2A0AA3A5EA7103C060F92DF665750E58301D0603551D0E0416041433795EB2" +
            "D84BFAA3F96E5930F64EC6A94C6FD36A300E0603551D0F0101FF040403020780" +
            "30130603551D25040C300A06082B0601050507030330770603551D1F0470306E" +
            "3035A033A031862F687474703A2F2F63726C332E64696769636572742E636F6D" +
            "2F736861322D617373757265642D63732D67312E63726C3035A033A031862F68" +
            "7474703A2F2F63726C342E64696769636572742E636F6D2F736861322D617373" +
            "757265642D63732D67312E63726C304C0603551D200445304330370609608648" +
            "0186FD6C0301302A302806082B06010505070201161C68747470733A2F2F7777" +
            "772E64696769636572742E636F6D2F4350533008060667810C01040130818406" +
            "082B0601050507010104783076302406082B060105050730018618687474703A" +
            "2F2F6F6373702E64696769636572742E636F6D304E06082B0601050507300286" +
            "42687474703A2F2F636163657274732E64696769636572742E636F6D2F446967" +
            "694365727453484132417373757265644944436F64655369676E696E6743412E" +
            "637274300C0603551D130101FF04023000300D06092A864886F70D01010B0500" +
            "038201010045B9D9868E494BD02635D0E42DDE80B37A865C389CFDD99BFC9B62" +
            "E2C169A73B5EABF282607439EFF5C61630886DEB63415B53683446A73041686C" +
            "326BA35FF0029FEF603D7C80FA0177A4DE35013529B01F759FD5041479BDBB6B" +
            "93B18144CB14E431BC144146848EF8ADB0E28952EAD1BB49E8547FFE99348170" +
            "36338B20C4E0B9D7C6A4E5BE3D57157F21904A5C864946313EA6B7D950EE0235" +
            "B5D2CD01490AD2B2A1AB5F66EC8986D64A1D9D239C131E09E5CA1C02A75F2D7E" +
            "C07E4C858856A6A58AB94DEAC8B3D3A5BBF492EE2463B156E6A0660BB452E359" +
            "22D00456F0DEE0ED15A8BF8FFF31008756B14EEE0AC14BCF19A3CD16819DC990" +
            "F5F45CDE21318201AE308201AA0201013081863072310B300906035504061302" +
            "555331153013060355040A130C446967694365727420496E6331193017060355" +
            "040B13107777772E64696769636572742E636F6D3131302F0603550403132844" +
            "696769436572742053484132204173737572656420494420436F646520536967" +
            "6E696E6720434102100D85090F3FACFF0A9008A12A9FB00A54300906052B0E03" +
            "021A0500300D06092A864886F70D01010B050004820100EAEEB9E1D4BFB979F1" +
            "A1C00EE1EC45069366CDD7489A0671F6DC9E3353F7FAEDCE7B87BD467ADFC850" +
            "877414966E7EB39C33367ABB03B3AA8BB1438BD952484CB807451499CAE8FDC9" +
            "527304D459D82CA039087560B5D3D0EA03DEA1B9472EFC44CBB55DD9A3C6A5C8" +
            "DFFD0786D5523F22604B412D6FC5A15E2D6285D7AB76EC216DE859391D129D51" +
            "6C27348EDAE7DC43335D12242D939CAF05385A118235F5B1E342EC034E70F655" +
            "793FF2FE037558EC2F45BD2683704F8FFD49B910131F4F2804B4282C5C36E41C" +
            "9E4E4F93446D44E3106760D265C5C7A849CF03426ACCB294712E51313D5414A7" +
            "8227AB79F6B18E2A2054E3FA781DAA2998EB33EDDCDA80").HexToByteArray();

        internal static readonly byte[] IndefiniteLengthContentDocument = (
            "3082265206092A864886F70D010702A08226433082263F020103310F300D0609" +
            "6086480165030402010500306806092A864886F70D010701A05B248004555665" +
            "7273696F6E3A310A0A322E31362E3834302E312E3130312E332E342E322E312D" +
            "486173683A49413134467074344A365A504F5A47506275364A676A524B696930" +
            "664E584A365A6B78746F744B665733383D0A0A0000A082106D308203C5308202" +
            "ADA003020102021002AC5C266A0B409B8F0B79F2AE462577300D06092A864886" +
            "F70D0101050500306C310B300906035504061302555331153013060355040A13" +
            "0C446967694365727420496E6331193017060355040B13107777772E64696769" +
            "636572742E636F6D312B30290603550403132244696769436572742048696768" +
            "204173737572616E636520455620526F6F74204341301E170D30363131313030" +
            "30303030305A170D3331313131303030303030305A306C310B30090603550406" +
            "1302555331153013060355040A130C446967694365727420496E633119301706" +
            "0355040B13107777772E64696769636572742E636F6D312B3029060355040313" +
            "2244696769436572742048696768204173737572616E636520455620526F6F74" +
            "20434130820122300D06092A864886F70D01010105000382010F003082010A02" +
            "82010100C6CCE573E6FBD4BBE52D2D32A6DFE5813FC9CD2549B6712AC3D59434" +
            "67A20A1CB05F69A640B1C4B7B28FD098A4A941593AD3DC94D63CDB7438A44ACC" +
            "4D2582F74AA5531238EEF3496D71917E63B6ABA65FC3A484F84F6251BEF8C5EC" +
            "DB3892E306E508910CC4284155FBCB5A89157E71E835BF4D72093DBE3A38505B" +
            "77311B8DB3C724459AA7AC6D00145A04B7BA13EB510A984141224E6561878141" +
            "50A6795C89DE194A57D52EE65D1C532C7E98CD1A0616A46873D03404135CA171" +
            "D35A7C55DB5E64E13787305604E511B4298012F1793988A202117C2766B788B7" +
            "78F2CA0AA838AB0A64C2BF665D9584C1A1251E875D1A500B2012CC41BB6E0B51" +
            "38B84BCB0203010001A3633061300E0603551D0F0101FF040403020186300F06" +
            "03551D130101FF040530030101FF301D0603551D0E04160414B13EC36903F8BF" +
            "4701D498261A0802EF63642BC3301F0603551D23041830168014B13EC36903F8" +
            "BF4701D498261A0802EF63642BC3300D06092A864886F70D0101050500038201" +
            "01001C1A0697DCD79C9F3C886606085721DB2147F82A67AABF183276401057C1" +
            "8AF37AD911658E35FA9EFC45B59ED94C314BB891E8432C8EB378CEDBE3537971" +
            "D6E5219401DA55879A2464F68A66CCDE9C37CDA834B1699B23C89E78222B7043" +
            "E35547316119EF58C5852F4E30F6A0311623C8E7E2651633CBBF1A1BA03DF8CA" +
            "5E8B318B6008892D0C065C52B7C4F90A98D1155F9F12BE7C366338BD44A47FE4" +
            "262B0AC497690DE98CE2C01057B8C876129155F24869D8BC2A025B0F44D42031" +
            "DBF4BA70265D90609EBC4B17092FB4CB1E4368C90727C1D25CF7EA21B968129C" +
            "3C9CBF9EFC805C9B63CDEC47AA252767A037F300827D54D7A9F8E92E13A377E8" +
            "1F4A308205E0308204C8A0030201020210070C57D60A8AB12F9B5F2D5CD2EDA5" +
            "04300D06092A864886F70D01010B0500306C310B300906035504061302555331" +
            "153013060355040A130C446967694365727420496E6331193017060355040B13" +
            "107777772E64696769636572742E636F6D312B30290603550403132244696769" +
            "4365727420455620436F6465205369676E696E6720434120285348413229301E" +
            "170D3137303531363030303030305A170D3230303532303132303030305A3081" +
            "F6311D301B060355040F0C1450726976617465204F7267616E697A6174696F6E" +
            "31133011060B2B0601040182373C0201031302555331193017060B2B06010401" +
            "82373C02010213084E657720596F726B3110300E060355040513073531303136" +
            "3333311E301C06035504091315353435205720456E6420417665204170742031" +
            "3645310E300C060355041113053130303234310B300906035504061302555331" +
            "0B3009060355040813024E593111300F060355040713084E657720596F726B31" +
            "1A3018060355040A13114F72656E204E6F766F746E792C204C4C43311A301806" +
            "0355040313114F72656E204E6F766F746E792C204C4C4330820122300D06092A" +
            "864886F70D01010105000382010F003082010A0282010100E253870D0B9B81C7" +
            "B2449CBD7BA9837EDFE23C79F54CE2C1BFA0F5C92C261AAC4E3D40CD890F8046" +
            "289C7BCF0525BC0D2E193C28E7AD3ABF87FD35CAB9CBC869CEF8964B80DD77CE" +
            "E209EFCC42A465FDC1338A478DAA80254D17679548ED78528A13632B66FD0531" +
            "555E04BCAF14A67AD266CECB03346FAB4757C1AC120FE34C15E4B5BBBDE3E601" +
            "EF9A8BBE6EE1802EF18E852233130B9DE0995FAACC869D414380B478E5E343AB" +
            "B2C18DDEC8170EB601301EC178B6CD66FF1E10D81637CB609FECCFBF91666090" +
            "A30F0D95B3BC204FDFC6314913C20091DBE60DEC88488C6B48E66E71247E11D2" +
            "7662A2A7106806BE8B95041F46D1375688F0B3C0BE805C750203010001A38201" +
            "F1308201ED301F0603551D230418301680148FE87EF06D326A000523C770976A" +
            "3A90FF6BEAD4301D0603551D0E0416041451B49AE070DEBDAE12B71C87F99F4E" +
            "0E495B30A1302E0603551D1104273025A02306082B06010505070803A0173015" +
            "0C1355532D4E455720594F524B2D35313031363333300E0603551D0F0101FF04" +
            "040302078030130603551D25040C300A06082B06010505070303307B0603551D" +
            "1F047430723037A035A0338631687474703A2F2F63726C332E64696769636572" +
            "742E636F6D2F4556436F64655369676E696E67534841322D67312E63726C3037" +
            "A035A0338631687474703A2F2F63726C342E64696769636572742E636F6D2F45" +
            "56436F64655369676E696E67534841322D67312E63726C304B0603551D200444" +
            "3042303706096086480186FD6C0302302A302806082B06010505070201161C68" +
            "747470733A2F2F7777772E64696769636572742E636F6D2F4350533007060567" +
            "810C0103307E06082B0601050507010104723070302406082B06010505073001" +
            "8618687474703A2F2F6F6373702E64696769636572742E636F6D304806082B06" +
            "010505073002863C687474703A2F2F636163657274732E64696769636572742E" +
            "636F6D2F44696769436572744556436F64655369676E696E6743412D53484132" +
            "2E637274300C0603551D130101FF04023000300D06092A864886F70D01010B05" +
            "0003820101000481103296C95A10AE7055ECB980F0A5910F756DB48704BB078C" +
            "B53E8859EDDF906E4B6207301C8BAE9CE39BE70D6D897EA2F32BD05270B8DC1E" +
            "44ADB0ECDCCAC7A858295E81E030B9A5CF571CD81536EBE36E4D87A9DD6E0C42" +
            "3B5E6F3A249C90AFB1418925F806C6B91346EBFDC61C1D392FA0861AE7B5ADF8" +
            "E0E213BBCCA53F826DA3B7514A80A7148968DC4855F584DBB2AB6E99569C6C89" +
            "333CFEEA0914F3C77A04E5EE44D7A27370F34FDFF4753762B62D55809C43FD63" +
            "F37BBB0239CB1B25F1F7C1932D0885F526CC87F589ED40BCEFBBDDD76620AA2D" +
            "5D373B9EE51F9FD02905EEC71DD5F5419010E5D061643A6A04211FA3165943AE" +
            "D0FF1FB58D23308206BC308205A4A003020102021003F1B4E15F3A82F1149678" +
            "B3D7D8475C300D06092A864886F70D01010B0500306C310B3009060355040613" +
            "02555331153013060355040A130C446967694365727420496E63311930170603" +
            "55040B13107777772E64696769636572742E636F6D312B302906035504031322" +
            "44696769436572742048696768204173737572616E636520455620526F6F7420" +
            "4341301E170D3132303431383132303030305A170D3237303431383132303030" +
            "305A306C310B300906035504061302555331153013060355040A130C44696769" +
            "4365727420496E6331193017060355040B13107777772E64696769636572742E" +
            "636F6D312B302906035504031322446967694365727420455620436F64652053" +
            "69676E696E672043412028534841322930820122300D06092A864886F70D0101" +
            "0105000382010F003082010A0282010100A753FA0FB2B513F164CF8480FCAE80" +
            "35D1B6D7C7A32CAC1A2CACF184AC3A35123A9291BA57E4C4C9F32FA8483CB7D6" +
            "6EDC9722BA517961AF432F0DB79BB44931AE44583EA4A196A7874F237EC36C65" +
            "2490553EA1CA237CC542E9C47A62459B7DDE6374CB9E6325F8849A9AAD454FAE" +
            "7D1FC813CB759BC9E1E18AF80B0C98F4CA3ED045AA7A1EA558933634BE2B2E2B" +
            "315866B432109F9DF052A1EFE83ED376F2405ADCFA6A3D1B4BAD76B08C5CEE36" +
            "BA83EA30A84CDEF10B2A584188AE0089AB03D11682202276EB5E54381262E1D2" +
            "7024DBED1F70D26409802DE2B69DCE1FF2BB21F36CDBD8B3197B8A509FEFEC36" +
            "0A5C9AB74AD308A03979FDDDBF3D3A09250203010001A3820358308203543012" +
            "0603551D130101FF040830060101FF020100300E0603551D0F0101FF04040302" +
            "018630130603551D25040C300A06082B06010505070303307F06082B06010505" +
            "07010104733071302406082B060105050730018618687474703A2F2F6F637370" +
            "2E64696769636572742E636F6D304906082B06010505073002863D687474703A" +
            "2F2F636163657274732E64696769636572742E636F6D2F446967694365727448" +
            "6967684173737572616E63654556526F6F7443412E63727430818F0603551D1F" +
            "0481873081843040A03EA03C863A687474703A2F2F63726C332E646967696365" +
            "72742E636F6D2F4469676943657274486967684173737572616E63654556526F" +
            "6F7443412E63726C3040A03EA03C863A687474703A2F2F63726C342E64696769" +
            "636572742E636F6D2F4469676943657274486967684173737572616E63654556" +
            "526F6F7443412E63726C308201C40603551D20048201BB308201B7308201B306" +
            "096086480186FD6C0302308201A4303A06082B06010505070201162E68747470" +
            "3A2F2F7777772E64696769636572742E636F6D2F73736C2D6370732D7265706F" +
            "7369746F72792E68746D3082016406082B06010505070202308201561E820152" +
            "0041006E007900200075007300650020006F0066002000740068006900730020" +
            "0043006500720074006900660069006300610074006500200063006F006E0073" +
            "007400690074007500740065007300200061006300630065007000740061006E" +
            "006300650020006F006600200074006800650020004400690067006900430065" +
            "00720074002000430050002F00430050005300200061006E0064002000740068" +
            "0065002000520065006C00790069006E00670020005000610072007400790020" +
            "00410067007200650065006D0065006E00740020007700680069006300680020" +
            "006C0069006D006900740020006C0069006100620069006C0069007400790020" +
            "0061006E0064002000610072006500200069006E0063006F00720070006F0072" +
            "0061007400650064002000680065007200650069006E00200062007900200072" +
            "00650066006500720065006E00630065002E301D0603551D0E041604148FE87E" +
            "F06D326A000523C770976A3A90FF6BEAD4301F0603551D23041830168014B13E" +
            "C36903F8BF4701D498261A0802EF63642BC3300D06092A864886F70D01010B05" +
            "00038201010019334A0C813337DBAD36C9E4C93ABBB51B2E7AA2E2F44342179E" +
            "BF4EA14DE1B1DBE981DD9F01F2E488D5E9FE09FD21C1EC5D80D2F0D6C143C2FE" +
            "772BDBF9D79133CE6CD5B2193BE62ED6C9934F88408ECDE1F57EF10FC6595672" +
            "E8EB6A41BD1CD546D57C49CA663815C1BFE091707787DCC98D31C90C29A233ED" +
            "8DE287CD898D3F1BFFD5E01A978B7CDA6DFBA8C6B23A666B7B01B3CDD8A634EC" +
            "1201AB9558A5C45357A860E6E70212A0B92364A24DBB7C81256421BECFEE4218" +
            "4397BBA53706AF4DFF26A54D614BEC4641B865CEB8799E08960B818C8A3B8FC7" +
            "998CA32A6E986D5E61C696B78AB9612D93B8EB0E0443D7F5FEA6F062D4996AA5" +
            "C1C1F06494803182154C30821548020103801451B49AE070DEBDAE12B71C87F9" +
            "9F4E0E495B30A1300D06096086480165030402010500A082015B301806092A86" +
            "4886F70D010903310B06092A864886F70D010701301C06092A864886F70D0109" +
            "05310F170D3138303632333139303131385A301E060B2A864886F70D01091002" +
            "10310F300D060B2A864886F70D0109100601302F06092A864886F70D01090431" +
            "220420F67BA3F4A22E1F7D27A0FF8940FD931FF3D5AAA6BA99B92B6ABA6B63B3" +
            "93E5DB3081CF060B2A864886F70D010910022F3181BF3081BC3081B93081B630" +
            "0B0609608648016503040201042017B677DFF2A3A70C935DCABCEAF6595183D6" +
            "2F8972E77AC688369EDE50D2B2563081843070A46E306C310B30090603550406" +
            "1302555331153013060355040A130C446967694365727420496E633119301706" +
            "0355040B13107777772E64696769636572742E636F6D312B3029060355040313" +
            "22446967694365727420455620436F6465205369676E696E6720434120285348" +
            "4132290210070C57D60A8AB12F9B5F2D5CD2EDA504300D06092A864886F70D01" +
            "010B050004820100AA697F9C5F4517BB139B8A7FAEC1BBBBAB53381B4F649EE1" +
            "9A35E9F5D47ABE0C2FCF881C6DC176E52F8191A43486630939264F42464B865D" +
            "B46979EF4996CD081CF615F194BBC9F539609000F2A2A6CBD9F88E497F4B907F" +
            "AEE95B55E73921CCAF89287FDC3F18FA1DB5D13BA7932A0559A9C4194E937611" +
            "003228D161B2EE40FD5C8BDB0463F90DFC5B0D0ABBBDFE7918BDDA2F5658F8E2" +
            "BA47362DC677CD8785D0AD76B73CC544411B7B168E471A449FDF8BD356719D56" +
            "C67E69F81F9534A2DE9D1CDD22717D3D3F84D6CB10C7FC2651945A000A2B7696" +
            "6A69CBE5D37B513E67E945B29CE2CC09CB2E4C8CDBAF105BCFE0927D14B8CEEF" +
            "92851AF4476053DCA18212AA308212A6060B2A864886F70D010910020E318212" +
            "953082129106092A864886F70D010702A08212823082127E020103310F300D06" +
            "09608648016503040201050030819C060B2A864886F70D0109100104A0818C04" +
            "818930818602010106096086480186FD6C07013031300D060960864801650304" +
            "0201050004208A42931DB9A7486014564D4922BE742D01B6ADF152B1421227B7" +
            "D6FBB943EEC202105B0A3E465D2092EB05B245401BC2594E180F323031383036" +
            "32333139303132305A02205718795810783FECCC7143D821251F44AEE16A57A1" +
            "CA24C1B2EF1796CB777AD3A0820F76308203B73082029FA00302010202100CE7" +
            "E0E517D846FE8FE560FC1BF03039300D06092A864886F70D0101050500306531" +
            "0B300906035504061302555331153013060355040A130C446967694365727420" +
            "496E6331193017060355040B13107777772E64696769636572742E636F6D3124" +
            "30220603550403131B4469676943657274204173737572656420494420526F6F" +
            "74204341301E170D3036313131303030303030305A170D333131313130303030" +
            "3030305A3065310B300906035504061302555331153013060355040A130C4469" +
            "67694365727420496E6331193017060355040B13107777772E64696769636572" +
            "742E636F6D312430220603550403131B44696769436572742041737375726564" +
            "20494420526F6F7420434130820122300D06092A864886F70D01010105000382" +
            "010F003082010A0282010100AD0E15CEE443805CB187F3B760F97112A5AEDC26" +
            "9488AAF4CEF520392858600CF880DAA9159532613CB5B128848A8ADC9F0A0C83" +
            "177A8F90AC8AE779535C31842AF60F98323676CCDEDD3CA8A2EF6AFB21F25261" +
            "DF9F20D71FE2B1D9FE1864D2125B5FF9581835BC47CDA136F96B7FD4B0383EC1" +
            "1BC38C33D9D82F18FE280FB3A783D6C36E44C061359616FE599C8B766DD7F1A2" +
            "4B0D2BFF0B72DA9E60D08E9035C678558720A1CFE56D0AC8497C3198336C22E9" +
            "87D0325AA2BA138211ED39179D993A72A1E6FAA4D9D5173175AE857D22AE3F01" +
            "4686F62879C8B1DAE45717C47E1C0EB0B492A656B3BDB297EDAAA7F0B7C5A83F" +
            "9516D0FFA196EB085F18774F0203010001A3633061300E0603551D0F0101FF04" +
            "0403020186300F0603551D130101FF040530030101FF301D0603551D0E041604" +
            "1445EBA2AFF492CB82312D518BA7A7219DF36DC80F301F0603551D2304183016" +
            "801445EBA2AFF492CB82312D518BA7A7219DF36DC80F300D06092A864886F70D" +
            "01010505000382010100A20EBCDFE2EDF0E372737A6494BFF77266D832E44275" +
            "62AE87EBF2D5D9DE56B39FCCCE1428B90D97605C124C58E4D33D834945589735" +
            "691AA847EA56C679AB12D8678184DF7F093C94E6B8262C20BD3DB32889F75FFF" +
            "22E297841FE965EF87E0DFC16749B35DEBB2092AEB26ED78BE7D3F2BF3B72635" +
            "6D5F8901B6495B9F01059BAB3D25C1CCB67FC2F16F86C6FA6468EB812D94EB42" +
            "B7FA8C1EDD62F1BE5067B76CBDF3F11F6B0C3607167F377CA95B6D7AF1124660" +
            "83D72704BE4BCE97BEC3672A6811DF80E70C3366BF130D146EF37F1F63101EFA" +
            "8D1B256D6C8FA5B76101B1D2A326A110719DADE2C3F9C39951B72B0708CE2EE6" +
            "50B2A7FA0A452FA2F0F23082053130820419A00302010202100AA125D6D6321B" +
            "7E41E405DA3697C215300D06092A864886F70D01010B05003065310B30090603" +
            "5504061302555331153013060355040A130C446967694365727420496E633119" +
            "3017060355040B13107777772E64696769636572742E636F6D31243022060355" +
            "0403131B4469676943657274204173737572656420494420526F6F7420434130" +
            "1E170D3136303130373132303030305A170D3331303130373132303030305A30" +
            "72310B300906035504061302555331153013060355040A130C44696769436572" +
            "7420496E6331193017060355040B13107777772E64696769636572742E636F6D" +
            "3131302F06035504031328446967694365727420534841322041737375726564" +
            "2049442054696D657374616D70696E6720434130820122300D06092A864886F7" +
            "0D01010105000382010F003082010A0282010100BDD032EE4BCD8F7FDDA9BA82" +
            "99C539542857B6234AC40E07453351107DD0F97D4D687EE7B6A0F48DB388E497" +
            "BF63219098BF13BC57D3C3E17E08D66A140038F72E1E3BEECCA6F63259FE5F65" +
            "3FE09BEBE34647061A557E0B277EC0A2F5A0E0DE223F0EFF7E95FBF3A3BA223E" +
            "18AC11E4F099036D3B857C09D3EE5DC89A0B54E3A809716BE0CF22100F75CF71" +
            "724E0AADDF403A5CB751E1A17914C64D2423305DBCEC3C606AAC2F07CCFDF0EA" +
            "47D988505EFD666E56612729898451E682E74650FD942A2CA7E4753EBA980F84" +
            "7F9F3114D6ADD5F264CB7B1E05D084197217F11706EF3DCDD64DEF0642FDA253" +
            "2A4F851DC41D3CAFCFDAAC10F5DDACACE956FF930203010001A38201CE308201" +
            "CA301D0603551D0E04160414F4B6E1201DFE29AED2E461A5B2A225B2C817356E" +
            "301F0603551D2304183016801445EBA2AFF492CB82312D518BA7A7219DF36DC8" +
            "0F30120603551D130101FF040830060101FF020100300E0603551D0F0101FF04" +
            "040302018630130603551D25040C300A06082B06010505070308307906082B06" +
            "010505070101046D306B302406082B060105050730018618687474703A2F2F6F" +
            "6373702E64696769636572742E636F6D304306082B0601050507300286376874" +
            "74703A2F2F636163657274732E64696769636572742E636F6D2F446967694365" +
            "7274417373757265644944526F6F7443412E6372743081810603551D1F047A30" +
            "78303AA038A0368634687474703A2F2F63726C342E64696769636572742E636F" +
            "6D2F4469676943657274417373757265644944526F6F7443412E63726C303AA0" +
            "38A0368634687474703A2F2F63726C332E64696769636572742E636F6D2F4469" +
            "676943657274417373757265644944526F6F7443412E63726C30500603551D20" +
            "044930473038060A6086480186FD6C000204302A302806082B06010505070201" +
            "161C68747470733A2F2F7777772E64696769636572742E636F6D2F435053300B" +
            "06096086480186FD6C0701300D06092A864886F70D01010B0500038201010071" +
            "9512E951875669CDEFDDDA7CAA637AB378CF06374084EF4B84BFCACF0302FDC5" +
            "A7C30E20422CAF77F32B1F0C215A2AB705341D6AAE99F827A266BF09AA60DF76" +
            "A43A930FF8B2D1D87C1962E85E82251EC4BA1C7B2C21E2D65B2C1435430468B2" +
            "DB7502E072C798D63C64E51F4810185F8938614D62462487638C91522CAF2989" +
            "E5781FD60B14A580D7124770B375D59385937EB69267FB536189A8F56B96C0F4" +
            "58690D7CC801B1B92875B7996385228C61CA79947E59FC8C0FE36FB50126B66C" +
            "A5EE875121E458609BBA0C2D2B6DA2C47EBBC4252B4702087C49AE13B6E17C42" +
            "4228C61856CF4134B6665DB6747BF55633222F2236B24BA24A95D8F5A68E5230" +
            "8206823082056AA003020102021009C0FC46C8044213B5598BAF284F4E41300D" +
            "06092A864886F70D01010B05003072310B300906035504061302555331153013" +
            "060355040A130C446967694365727420496E6331193017060355040B13107777" +
            "772E64696769636572742E636F6D3131302F0603550403132844696769436572" +
            "74205348413220417373757265642049442054696D657374616D70696E672043" +
            "41301E170D3137303130343030303030305A170D323830313138303030303030" +
            "5A304C310B30090603550406130255533111300F060355040A13084469676943" +
            "657274312A302806035504031321446967694365727420534841322054696D65" +
            "7374616D7020526573706F6E64657230820122300D06092A864886F70D010101" +
            "05000382010F003082010A02820101009E95986A343B731BA87EFCC7BE296989" +
            "C76826465F3D8D62738781A3A19CF0B75B24375A92D4F459D77689E4DCD527F0" +
            "D566BC0AEEB42B3167AC58C54A91592B451E0901D664B359EE8D664DFB235ECC" +
            "100D0B8A67EF52AEA00890C252F7F5A8B56E9B2C7B9DE7B53EFB78CD325018BF" +
            "40B54C8CBB57F4A04F11456C4242B9E5AFD6DFF4A77C0A68960FD25F2957CEFB" +
            "1D32FFF411A11322FB12CBEFD753D2EB97CBA2AC1B1D9D58215182C2C2DEEA2B" +
            "3F2C2284D043EC3B3B3F47C4F656DC453798B46B74B559AF785769C80F090278" +
            "DDD853C199DB60C49DEAAEAFE07E864A5CA95861A85E748A012868724EA7869D" +
            "B5025287706648D38EEF8124CCDCD8650203010001A382033830820334300E06" +
            "03551D0F0101FF040403020780300C0603551D130101FF040230003016060355" +
            "1D250101FF040C300A06082B06010505070308308201BF0603551D20048201B6" +
            "308201B2308201A106096086480186FD6C070130820192302806082B06010505" +
            "070201161C68747470733A2F2F7777772E64696769636572742E636F6D2F4350" +
            "533082016406082B06010505070202308201561E8201520041006E0079002000" +
            "75007300650020006F0066002000740068006900730020004300650072007400" +
            "6900660069006300610074006500200063006F006E0073007400690074007500" +
            "740065007300200061006300630065007000740061006E006300650020006F00" +
            "6600200074006800650020004400690067006900430065007200740020004300" +
            "50002F00430050005300200061006E0064002000740068006500200052006500" +
            "6C00790069006E00670020005000610072007400790020004100670072006500" +
            "65006D0065006E00740020007700680069006300680020006C0069006D006900" +
            "740020006C0069006100620069006C00690074007900200061006E0064002000" +
            "610072006500200069006E0063006F00720070006F0072006100740065006400" +
            "2000680065007200650069006E00200062007900200072006500660065007200" +
            "65006E00630065002E300B06096086480186FD6C0315301F0603551D23041830" +
            "168014F4B6E1201DFE29AED2E461A5B2A225B2C817356E301D0603551D0E0416" +
            "0414E1A7324AEE0121287D54D5F207926EB4070F3D8730710603551D1F046A30" +
            "683032A030A02E862C687474703A2F2F63726C332E64696769636572742E636F" +
            "6D2F736861322D617373757265642D74732E63726C3032A030A02E862C687474" +
            "703A2F2F63726C342E64696769636572742E636F6D2F736861322D6173737572" +
            "65642D74732E63726C30818506082B0601050507010104793077302406082B06" +
            "0105050730018618687474703A2F2F6F6373702E64696769636572742E636F6D" +
            "304F06082B060105050730028643687474703A2F2F636163657274732E646967" +
            "69636572742E636F6D2F44696769436572745348413241737375726564494454" +
            "696D657374616D70696E6743412E637274300D06092A864886F70D01010B0500" +
            "03820101001EF0418232AEEDF1B43513DC50C2D597AE22229D0E0EAF33D34CFD" +
            "7CBF6F0111A79465225CC622A1C889526B9A8C735CD95E3F32DE16604C8B36FD" +
            "31990ABDC184B78D1DEF8926130556F347CD475BAD84B238AF6A23B545E31E88" +
            "324680D2B7A69922FDC178CFF58BD80C8C0509EE44E680D56D70CC9F531E27DD" +
            "2A48DEDA9365AD6E65A399A7C2400E73CC584F8F4528E5BC9C88E628CE605D2D" +
            "255D8B732EA50D5B51DA9A4EFF50058928DAF278BBD258788D44A7AC3A009178" +
            "69896404D35D96DF2ABFF9A54C2C93FFE68ADD82ACF1D2B3A2869AC15589566A" +
            "473FFAD6339543358905785A3A69DA22B80443D36F6835367A143E45E9986486" +
            "0F130C264A3182024D308202490201013081863072310B300906035504061302" +
            "555331153013060355040A130C446967694365727420496E6331193017060355" +
            "040B13107777772E64696769636572742E636F6D3131302F0603550403132844" +
            "69676943657274205348413220417373757265642049442054696D657374616D" +
            "70696E67204341021009C0FC46C8044213B5598BAF284F4E41300D0609608648" +
            "0165030402010500A08198301A06092A864886F70D010903310D060B2A864886" +
            "F70D0109100104301C06092A864886F70D010905310F170D3138303632333139" +
            "303132305A302F06092A864886F70D01090431220420DAE824F466CED9AD07AC" +
            "66897D7489E25E678911AA3E85BBBAB76F93DF77FFCA302B060B2A864886F70D" +
            "010910020C311C301A301830160414400191475C98891DEBA104AF47091B5EB6" +
            "D4CBCB300D06092A864886F70D010101050004820100451DCB791B5E31865B9F" +
            "697B8A790C4DB1867261B10EDCEB65F8AD03350D8D25D89012E8A9C46D3DA436" +
            "04F222C645A8C4C7658D0C5C31A473A09FE314D049A886001CAC3C2C5B106BD7" +
            "1D02785A3BD02E4171E70A9AA1C7EC5BDB7D1B5A2F3A5F6EF92EE3296CB8FC52" +
            "D7ABF3ADB91F2BF30A2D0DE9713CFE5CF203B6D3F0F51897DCB075E8AAB8CB96" +
            "1F9505656CA73C8935E29A71C09BE3AA0BA1326FA295B6686771D6431BB2DAB3" +
            "8DECE164DE7487945C09DC0BF525086256C4D22CC71C70DD886BDD2EEF609631" +
            "771EF1E49426819679F9C080815B27AD93090D56DBF566D585693B27B5FDA70B" +
            "18401534B87B522B7B0AA98E3F31DFDD64FDA0DE9471").HexToByteArray();
    }
}
