// Copyright 2019 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License"); you may not
// use this file except in compliance with the License. You may obtain a copy of
// the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
// WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
// License for the specific language governing permissions and limitations under
// the License.

//go:build windows
// +build windows

package attest

// Error codes.
var (
	isReadyErrors = map[uint32]string{
		0x00000002: "Platform restart is required (shutdown).",
		0x00000004: "Platform restart is required (reboot).",
		0x00000008: "The TPM is already owned.",
		0x00000010: "Physical presence is required to provision the TPM.",
		0x00000020: "The TPM is disabled or deactivated.",
		0x00000040: "TPM ownership was taken.",
		0x00000080: "An endorsement key exists in the TPM.",
		0x00000100: "The TPM owner authorization is not properly stored in the registry.",
		0x00000200: "The Storage Root Key (SRK) authorization value is not all zeros.",
		0x00000800: "The operating system's registry information about the TPM’s Storage Root Key does not match the TPM Storage Root Key.",
		0x00001000: "The TPM permanent flag to allow reading of the Storage Root Key public value is not set.",
		0x00002000: "The monotonic counter incremented during boot has not been created.",
		0x00020000: "Windows Group Policy is configured to not store any TPM owner authorization so the TPM cannot be fully ready.",
		0x00040000: "The EK Certificate was not read from the TPM NV Ram and stored in the registry.",
		0x00080000: "The TCG event log is empty or cannot be read.",
		0x00100000: "The TPM is not owned.",
		0x00200000: "An error occurred, but not specific to a particular task.",
		0x00400000: "The device lock counter has not been created.",
		0x00800000: "The device identifier has not been created.",
	}
	tpmErrNums = map[uint32]struct {
		name        string
		description string
	}{
		0x80280001: {"TPM_E_AUTHFAIL", "Authentication failed."},
		0x80280002: {"TPM_E_BADINDEX", "The index to a PCR, DIR or other register is incorrect."},
		0x80280003: {"TPM_E_BAD_PARAMETER", "One or more parameter is bad."},
		0x80280004: {"TPM_E_AUDITFAILURE", "An operation completed successfully but the auditing of that operation failed."},
		0x80280005: {"TPM_E_CLEAR_DISABLED", "The clear disable flag is set and all clear operations now require physical access."},
		0x80280006: {"TPM_E_DEACTIVATED", "Activate the Trusted Platform Module (TPM)."},
		0x80280007: {"TPM_E_DISABLED", "Enable the Trusted Platform Module (TPM)."},
		0x80280008: {"TPM_E_DISABLED_CMD", "The target command has been disabled."},
		0x80280009: {"TPM_E_FAIL", "The operation failed."},
		0x8028000A: {"TPM_E_BAD_ORDINAL", "The ordinal was unknown or inconsistent."},
		0x8028000B: {"TPM_E_INSTALL_DISABLED", "The ability to install an owner is disabled."},
		0x8028000C: {"TPM_E_INVALID_KEYHANDLE", "The key handle cannot be interpreted."},
		0x8028000D: {"TPM_E_KEYNOTFOUND", "The key handle points to an invalid key."},
		0x8028000E: {"TPM_E_INAPPROPRIATE_ENC", "Unacceptable encryption scheme."},
		0x8028000F: {"TPM_E_MIGRATEFAIL", "Migration authorization failed."},
		0x80280010: {"TPM_E_INVALID_PCR_INFO", "PCR information could not be interpreted."},
		0x80280011: {"TPM_E_NOSPACE", "No room to load key."},
		0x80280012: {"TPM_E_NOSRK", "There is no Storage Root Key (SRK) set."},
		0x80280013: {"TPM_E_NOTSEALED_BLOB", "An encrypted blob is invalid or was not created by this TPM."},
		0x80280014: {"TPM_E_OWNER_SET", "The Trusted Platform Module (TPM) already has an owner."},
		0x80280015: {"TPM_E_RESOURCES", "The TPM has insufficient internal resources to perform the requested action."},
		0x80280016: {"TPM_E_SHORTRANDOM", "A random string was too short."},
		0x80280017: {"TPM_E_SIZE", "The TPM does not have the space to perform the operation."},
		0x80280018: {"TPM_E_WRONGPCRVAL", "The named PCR value does not match the current PCR value."},
		0x80280019: {"TPM_E_BAD_PARAM_SIZE", "The paramSize argument to the command has the incorrect value ."},
		0x8028001A: {"TPM_E_SHA_THREAD", "There is no existing SHA-1 thread."},
		0x8028001B: {"TPM_E_SHA_ERROR", "The calculation is unable to proceed because the existing SHA-1 thread has already encountered an error."},
		0x8028001C: {"TPM_E_FAILEDSELFTEST", "The TPM hardware device reported a failure during its internal self test. Try restarting the computer to resolve the problem. If the problem continues, you might need to replace your TPM hardware or motherboard."},
		0x8028001D: {"TPM_E_AUTH2FAIL", "The authorization for the second key in a 2 key function failed authorization."},
		0x8028001E: {"TPM_E_BADTAG", "The tag value sent to for a command is invalid."},
		0x8028001F: {"TPM_E_IOERROR", "An IO error occurred transmitting information to the TPM."},
		0x80280020: {"TPM_E_ENCRYPT_ERROR", "The encryption process had a problem."},
		0x80280021: {"TPM_E_DECRYPT_ERROR", "The decryption process did not complete."},
		0x80280022: {"TPM_E_INVALID_AUTHHANDLE", "An invalid handle was used."},
		0x80280023: {"TPM_E_NO_ENDORSEMENT", "The TPM does not have an Endorsement Key (EK) installed."},
		0x80280024: {"TPM_E_INVALID_KEYUSAGE", "The usage of a key is not allowed."},
		0x80280025: {"TPM_E_WRONG_ENTITYTYPE", "The submitted entity type is not allowed."},
		0x80280026: {"TPM_E_INVALID_POSTINIT", "The command was received in the wrong sequence relative to TPM_Init and a subsequent TPM_Startup."},
		0x80280027: {"TPM_E_INAPPROPRIATE_SIG", "Signed data cannot include additional DER information."},
		0x80280028: {"TPM_E_BAD_KEY_PROPERTY", "The key properties in TPM_KEY_PARMs are not supported by this TPM."},
		0x80280029: {"TPM_E_BAD_MIGRATION", "The migration properties of this key are incorrect."},
		0x8028002A: {"TPM_E_BAD_SCHEME", "The signature or encryption scheme for this key is incorrect or not permitted in this situation."},
		0x8028002B: {"TPM_E_BAD_DATASIZE", "The size of the data (or blob) parameter is bad or inconsistent with the referenced key."},
		0x8028002C: {"TPM_E_BAD_MODE", "A mode parameter is bad, such as capArea or subCapArea for TPM_GetCapability, phsicalPresence parameter for TPM_PhysicalPresence, or migrationType for TPM_CreateMigrationBlob."},
		0x8028002D: {"TPM_E_BAD_PRESENCE", "Either the physicalPresence or physicalPresenceLock bits have the wrong value."},
		0x8028002E: {"TPM_E_BAD_VERSION", "The TPM cannot perform this version of the capability."},
		0x8028002F: {"TPM_E_NO_WRAP_TRANSPORT", "The TPM does not allow for wrapped transport sessions."},
		0x80280030: {"TPM_E_AUDITFAIL_UNSUCCESSFUL", "TPM audit construction failed and the underlying command was returning a failure code also."},
		0x80280031: {"TPM_E_AUDITFAIL_SUCCESSFUL", "TPM audit construction failed and the underlying command was returning success."},
		0x80280032: {"TPM_E_NOTRESETABLE", "Attempt to reset a PCR register that does not have the resettable attribute."},
		0x80280033: {"TPM_E_NOTLOCAL", "Attempt to reset a PCR register that requires locality and locality modifier not part of command transport."},
		0x80280034: {"TPM_E_BAD_TYPE", "Make identity blob not properly typed."},
		0x80280035: {"TPM_E_INVALID_RESOURCE", "When saving context identified resource type does not match actual resource."},
		0x80280036: {"TPM_E_NOTFIPS", "The TPM is attempting to execute a command only available when in FIPS mode."},
		0x80280037: {"TPM_E_INVALID_FAMILY", "The command is attempting to use an invalid family ID."},
		0x80280038: {"TPM_E_NO_NV_PERMISSION", "The permission to manipulate the NV storage is not available."},
		0x80280039: {"TPM_E_REQUIRES_SIGN", "The operation requires a signed command."},
		0x8028003A: {"TPM_E_KEY_NOTSUPPORTED", "Wrong operation to load an NV key."},
		0x8028003B: {"TPM_E_AUTH_CONFLICT", "NV_LoadKey blob requires both owner and blob authorization."},
		0x8028003C: {"TPM_E_AREA_LOCKED", "The NV area is locked and not writtable."},
		0x8028003D: {"TPM_E_BAD_LOCALITY", "The locality is incorrect for the attempted operation."},
		0x8028003E: {"TPM_E_READ_ONLY", "The NV area is read only and can't be written to."},
		0x8028003F: {"TPM_E_PER_NOWRITE", "There is no protection on the write to the NV area."},
		0x80280040: {"TPM_E_FAMILYCOUNT", "The family count value does not match."},
		0x80280041: {"TPM_E_WRITE_LOCKED", "The NV area has already been written to."},
		0x80280042: {"TPM_E_BAD_ATTRIBUTES", "The NV area attributes conflict."},
		0x80280043: {"TPM_E_INVALID_STRUCTURE", "The structure tag and version are invalid or inconsistent."},
		0x80280044: {"TPM_E_KEY_OWNER_CONTROL", "The key is under control of the TPM Owner and can only be evicted by the TPM Owner."},
		0x80280045: {"TPM_E_BAD_COUNTER", "The counter handle is incorrect."},
		0x80280046: {"TPM_E_NOT_FULLWRITE", "The write is not a complete write of the area."},
		0x80280047: {"TPM_E_CONTEXT_GAP", "The gap between saved context counts is too large."},
		0x80280048: {"TPM_E_MAXNVWRITES", "The maximum number of NV writes without an owner has been exceeded."},
		0x80280049: {"TPM_E_NOOPERATOR", "No operator AuthData value is set."},
		0x8028004A: {"TPM_E_RESOURCEMISSING", "The resource pointed to by context is not loaded."},
		0x8028004B: {"TPM_E_DELEGATE_LOCK", "The delegate administration is locked."},
		0x8028004C: {"TPM_E_DELEGATE_FAMILY", "Attempt to manage a family other than the delegated family."},
		0x8028004D: {"TPM_E_DELEGATE_ADMIN", "Delegation table management not enabled."},
		0x8028004E: {"TPM_E_TRANSPORT_NOTEXCLUSIVE", "There was a command executed outside of an exclusive transport session."},
		0x8028004F: {"TPM_E_OWNER_CONTROL", "Attempt to context save an owner evict controlled key."},
		0x80280050: {"TPM_E_DAA_RESOURCES", "The DAA command has no resources available to execute the command."},
		0x80280051: {"TPM_E_DAA_INPUT_DATA0", "The consistency check on DAA parameter inputData0 has failed."},
		0x80280052: {"TPM_E_DAA_INPUT_DATA1", "The consistency check on DAA parameter inputData1 has failed."},
		0x80280053: {"TPM_E_DAA_ISSUER_SETTINGS", "The consistency check on DAA_issuerSettings has failed."},
		0x80280054: {"TPM_E_DAA_TPM_SETTINGS", "The consistency check on DAA_tpmSpecific has failed."},
		0x80280055: {"TPM_E_DAA_STAGE", "The atomic process indicated by the submitted DAA command is not the expected process."},
		0x80280056: {"TPM_E_DAA_ISSUER_VALIDITY", "The issuer's validity check has detected an inconsistency."},
		0x80280057: {"TPM_E_DAA_WRONG_W", "The consistency check on w has failed."},
		0x80280058: {"TPM_E_BAD_HANDLE", "The handle is incorrect."},
		0x80280059: {"TPM_E_BAD_DELEGATE", "Delegation is not correct."},
		0x8028005A: {"TPM_E_BADCONTEXT", "The context blob is invalid."},
		0x8028005B: {"TPM_E_TOOMANYCONTEXTS", "Too many contexts held by the TPM."},
		0x8028005C: {"TPM_E_MA_TICKET_SIGNATURE", "Migration authority signature validation failure."},
		0x8028005D: {"TPM_E_MA_DESTINATION", "Migration destination not authenticated."},
		0x8028005E: {"TPM_E_MA_SOURCE", "Migration source incorrect."},
		0x8028005F: {"TPM_E_MA_AUTHORITY", "Incorrect migration authority."},
		0x80280061: {"TPM_E_PERMANENTEK", "Attempt to revoke the EK and the EK is not revocable."},
		0x80280062: {"TPM_E_BAD_SIGNATURE", "Bad signature of CMK ticket."},
		0x80280063: {"TPM_E_NOCONTEXTSPACE", "There is no room in the context list for additional contexts."},
		0x80280400: {"TPM_E_COMMAND_BLOCKED", "The command was blocked."},
		0x80280401: {"TPM_E_INVALID_HANDLE", "The specified handle was not found."},
		0x80280402: {"TPM_E_DUPLICATE_VHANDLE", "The TPM returned a duplicate handle and the command needs to be resubmitted."},
		0x80280403: {"TPM_E_EMBEDDED_COMMAND_BLOCKED", "The command within the transport was blocked."},
		0x80280404: {"TPM_E_EMBEDDED_COMMAND_UNSUPPORTED", "The command within the transport is not supported."},
		0x80280800: {"TPM_E_RETRY", "The TPM is too busy to respond to the command immediately, but the command could be resubmitted at a later time."},
		0x80280801: {"TPM_E_NEEDS_SELFTEST", "SelfTestFull has not been run."},
		0x80280802: {"TPM_E_DOING_SELFTEST", "The TPM is currently executing a full selftest."},
		0x80280803: {"TPM_E_DEFEND_LOCK_RUNNING", "The TPM is defending against dictionary attacks and is in a time-out period."},
		0x80280082: {"TPM_20_E_ATTRIBUTES", "TPM 2.0: Inconsistent attributes."},
		0x80280083: {"TPM_20_E_HASH", "TPM 2.0: Hash algorithm not supported or not appropriate."},
		0x80280084: {"TPM_20_E_VALUE", "TPM 2.0: Value is out of range or is not correct for the context."},
		0x80280085: {"TPM_20_E_HIERARCHY", "TPM 2.0: Hierarchy is not enabled or is not correct for the use."},
		0x80280086: {"TPM_20_E_KEY_SIZE", "TPM 2.0: Key size is not supported."},
		0x80280087: {"TPM_20_E_MGF", "TPM 2.0: Mask generation function not supported."},
		0x80280089: {"TPM_20_E_MODE", "TPM 2.0: Mode of operation not supported."},
		0x8028008A: {"TPM_20_E_TYPE", "TPM 2.0: The type of the value is not appropriate for the use."},
		0x8028008B: {"TPM_20_E_HANDLE", "TPM 2.0: The Handle is not correct for the use."},
		0x8028008C: {"TPM_20_E_KDF", "TPM 2.0: Unsupported key derivation function or function not appropriate for use."},
		0x8028008D: {"TPM_20_E_RANGE", "TPM 2.0: Value was out of allowed range."},
		0x8028008E: {"TPM_20_E_AUTH_FAIL", "TPM 2.0: The authorization HMAC check failed and DA counter incremented."},
		0x8028008F: {"TPM_20_E_NONCE", "TPM 2.0: Invalid nonce size."},
		0x80280092: {"TPM_20_E_SCHEME", "TPM 2.0: Unsupported or incompatible scheme."},
		0x80280095: {"TPM_20_E_SIZE", "TPM 2.0: Structure is wrong size.."},
		0x80280096: {"TPM_20_E_SYMMETRIC", "TPM 2.0: Unsupported symmetric algorithm or key size, or not appropriate for instance."},
		0x80280097: {"TPM_20_E_TAG", "TPM 2.0: Incorrect structure tag."},
		0x80280098: {"TPM_20_E_SELECTOR", "TPM 2.0: Union selector is incorrect."},
		0x8028009A: {"TPM_20_E_INSUFFICIENT", "TPM 2.0: The TPM was unable to unmarshal a value because there were not enough octets in the input buffer."},
		0x8028009B: {"TPM_20_E_SIGNATURE", "TPM 2.0: The signature is not valid."},
		0x8028009C: {"TPM_20_E_KEY", "TPM 2.0: Key fields are not compatible with the selected use."},
		0x8028009D: {"TPM_20_E_POLICY_FAIL", "TPM 2.0: A policy check failed."},
		0x8028009F: {"TPM_20_E_INTEGRITY", "TPM 2.0: Integrity check failed."},
		0x802800A0: {"TPM_20_E_TICKET", "TPM 2.0: Invalid ticket."},
		0x802800A1: {"TPM_20_E_RESERVED_BITS", "TPM 2.0: Reserved bits not set to zero as required."},
		0x802800A2: {"TPM_20_E_BAD_AUTH", "TPM 2.0: Authorization failure without DA implications."},
		0x802800A3: {"TPM_20_E_EXPIRED", "TPM 2.0: The policy has expired."},
		0x802800A4: {"TPM_20_E_POLICY_CC", "TPM 2.0: The command code in the policy is not the command code of the command or the command code in a policy command references a command that is not implemented."},
		0x802800A5: {"TPM_20_E_BINDING", "TPM 2.0: Public and sensitive portions of an object are not cryptographically bound."},
		0x802800A6: {"TPM_20_E_CURVE", "TPM 2.0: Curve not supported."},
		0x802800A7: {"TPM_20_E_ECC_POINT", "TPM 2.0: Point is not on the required curve."},
		0x80280100: {"TPM_20_E_INITIALIZE", "TPM 2.0: TPM not initialized."},
		0x80280101: {"TPM_20_E_FAILURE", "TPM 2.0: Commands not being accepted because of a TPM failure."},
		0x80280103: {"TPM_20_E_SEQUENCE", "TPM 2.0: Improper use of a sequence handle."},
		0x8028010B: {"TPM_20_E_PRIVATE", "TPM 2.0: TPM_RC_PRIVATE error."},
		0x80280119: {"TPM_20_E_HMAC", "TPM 2.0: TPM_RC_HMAC."},
		0x80280120: {"TPM_20_E_DISABLED", "TPM 2.0: TPM_RC_DISABLED."},
		0x80280121: {"TPM_20_E_EXCLUSIVE", "TPM 2.0: Command failed because audit sequence required exclusivity."},
		0x80280123: {"TPM_20_E_ECC_CURVE", "TPM 2.0: Unsupported ECC curve."},
		0x80280124: {"TPM_20_E_AUTH_TYPE", "TPM 2.0: Authorization handle is not correct for command."},
		0x80280125: {"TPM_20_E_AUTH_MISSING", "TPM 2.0: Command requires an authorization session for handle and is not present."},
		0x80280126: {"TPM_20_E_POLICY", "TPM 2.0: Policy failure in Math Operation or an invalid authPolicy value."},
		0x80280127: {"TPM_20_E_PCR", "TPM 2.0: PCR check fail."},
		0x80280128: {"TPM_20_E_PCR_CHANGED", "TPM 2.0: PCR have changed since checked."},
		0x8028012D: {"TPM_20_E_UPGRADE", "TPM 2.0: The TPM is not in the right mode for upgrade."},
		0x8028012E: {"TPM_20_E_TOO_MANY_CONTEXTS", "TPM 2.0: Context ID counter is at maximum."},
		0x8028012F: {"TPM_20_E_AUTH_UNAVAILABLE", "TPM 2.0: authValue or authPolicy is not available for selected entity."},
		0x80280130: {"TPM_20_E_REBOOT", "TPM 2.0: A _TPM_Init and Startup(CLEAR) is required before the TPM can resume operation."},
		0x80280131: {"TPM_20_E_UNBALANCED", "TPM 2.0: The protection algorithms (hash and symmetric) are not reasonably balanced. The digest size of the hash must be larger than the key size of the symmetric algorithm."},
		0x80280142: {"TPM_20_E_COMMAND_SIZE", "TPM 2.0: The TPM command's commandSize value is inconsistent with contents of the command buffer; either the size is not the same as the bytes loaded by the hardware interface layer or the value is not large enough to hold a command header."},
		0x80280143: {"TPM_20_E_COMMAND_CODE", "TPM 2.0: Command code not supported."},
		0x80280144: {"TPM_20_E_AUTHSIZE", "TPM 2.0: The value of authorizationSize is out of range or the number of octets in the authorization Area is greater than required."},
		0x80280145: {"TPM_20_E_AUTH_CONTEXT", "TPM 2.0: Use of an authorization session with a context command or another command that cannot have an authorization session."},
		0x80280146: {"TPM_20_E_NV_RANGE", "TPM 2.0: NV offset+size is out of range."},
		0x80280147: {"TPM_20_E_NV_SIZE", "TPM 2.0: Requested allocation size is larger than allowed."},
		0x80280148: {"TPM_20_E_NV_LOCKED", "TPM 2.0: NV access locked."},
		0x80280149: {"TPM_20_E_NV_AUTHORIZATION", "TPM 2.0: NV access authorization fails in command actions."},
		0x8028014A: {"TPM_20_E_NV_UNINITIALIZED", "TPM 2.0: An NV index is used before being initialized or the state saved by TPM2_Shutdown(STATE) could not be restored."},
		0x8028014B: {"TPM_20_E_NV_SPACE", "TPM 2.0: Insufficient space for NV allocation."},
		0x8028014C: {"TPM_20_E_NV_DEFINED", "TPM 2.0: NV index or persistent object already defined."},
		0x80280150: {"TPM_20_E_BAD_CONTEXT", "TPM 2.0: Context in TPM2_ContextLoad() is not valid."},
		0x80280151: {"TPM_20_E_CPHASH", "TPM 2.0: chHash value already set or not correct for use."},
		0x80280152: {"TPM_20_E_PARENT", "TPM 2.0: Handle for parent is not a valid parent."},
		0x80280153: {"TPM_20_E_NEEDS_TEST", "TPM 2.0: Some function needs testing."},
		0x80280154: {"TPM_20_E_NO_RESULT", "TPM 2.0: returned when an internal function cannot process a request due to an unspecified problem. This code is usually related to invalid parameters that are not properly filtered by the input unmarshaling code."},
		0x80280155: {"TPM_20_E_SENSITIVE", "TPM 2.0: The sensitive area did not unmarshal correctly after decryption - this code is used in lieu of the other unmarshaling errors so that an attacker cannot determine where the unmarshaling error occurred."},
		0x80280901: {"TPM_20_E_CONTEXT_GAP", "TPM 2.0: Gap for context ID is too large."},
		0x80280902: {"TPM_20_E_OBJECT_MEMORY", "TPM 2.0: Out of memory for object contexts."},
		0x80280903: {"TPM_20_E_SESSION_MEMORY", "TPM 2.0: Out of memory for session contexts."},
		0x80280904: {"TPM_20_E_MEMORY", "TPM 2.0: Out of shared object/session memory or need space for internal operations."},
		0x80280905: {"TPM_20_E_SESSION_HANDLES", "TPM 2.0: Out of session handles - a session must be flushed before a new session may be created."},
		0x80280906: {"TPM_20_E_OBJECT_HANDLES", "TPM 2.0: Out of object handles - the handle space for objects is depleted and a reboot is required."},
		0x80280907: {"TPM_20_E_LOCALITY", "TPM 2.0: Bad locality."},
		0x80280908: {"TPM_20_E_YIELDED", "TPM 2.0: The TPM has suspended operation on the command; forward progress was made and the command may be retried."},
		0x80280909: {"TPM_20_E_CANCELED", "TPM 2.0: The command was canceled."},
		0x8028090A: {"TPM_20_E_TESTING", "TPM 2.0: TPM is performing self-tests."},
		0x80280920: {"TPM_20_E_NV_RATE", "TPM 2.0: The TPM is rate-limiting accesses to prevent wearout of NV."},
		0x80280921: {"TPM_20_E_LOCKOUT", "TPM 2.0: Authorization for objects subject to DA protection are not allowed at this time because the TPM is in DA lockout mode."},
		0x80280922: {"TPM_20_E_RETRY", "TPM 2.0: The TPM was not able to start the command."},
		0x80280923: {"TPM_20_E_NV_UNAVAILABLE", "TPM 2.0: the command may require writing of NV and NV is not current accessible.."},
		0x80284001: {"TBS_E_INTERNAL_ERROR", "An internal software error has been detected."},
		0x80284002: {"TBS_E_BAD_PARAMETER", "One or more input parameters is bad."},
		0x80284003: {"TBS_E_INVALID_OUTPUT_POINTER", "A specified output pointer is bad."},
		0x80284004: {"TBS_E_INVALID_CONTEXT", "The specified context handle does not refer to a valid context."},
		0x80284005: {"TBS_E_INSUFFICIENT_BUFFER", "A specified output buffer is too small."},
		0x80284006: {"TBS_E_IOERROR", "An error occurred while communicating with the TPM."},
		0x80284007: {"TBS_E_INVALID_CONTEXT_PARAM", "One or more context parameters is invalid."},
		0x80284008: {"TBS_E_SERVICE_NOT_RUNNING", "The TBS service is not running and could not be started."},
		0x80284009: {"TBS_E_TOO_MANY_TBS_CONTEXTS", "A new context could not be created because there are too many open contexts."},
		0x8028400A: {"TBS_E_TOO_MANY_RESOURCES", "A new virtual resource could not be created because there are too many open virtual resources."},
		0x8028400B: {"TBS_E_SERVICE_START_PENDING", "The TBS service has been started but is not yet running."},
		0x8028400C: {"TBS_E_PPI_NOT_SUPPORTED", "The physical presence interface is not supported."},
		0x8028400D: {"TBS_E_COMMAND_CANCELED", "The command was canceled."},
		0x8028400E: {"TBS_E_BUFFER_TOO_LARGE", "The input or output buffer is too large."},
		0x8028400F: {"TBS_E_TPM_NOT_FOUND", "A compatible Trusted Platform Module (TPM) Security Device cannot be found on this computer."},
		0x80284010: {"TBS_E_SERVICE_DISABLED", "The TBS service has been disabled."},
		0x80284011: {"TBS_E_NO_EVENT_LOG", "No TCG event log is available."},
		0x80284012: {"TBS_E_ACCESS_DENIED", "The caller does not have the appropriate rights to perform the requested operation."},
		0x80284013: {"TBS_E_PROVISIONING_NOT_ALLOWED", "The TPM provisioning action is not allowed by the specified flags. For provisioning to be successful, one of several actions may be required. The TPM management console (tpm.msc) action to make the TPM Ready may help. For further information, see the documentation for the Win32_Tpm WMI method 'Provision'. (The actions that may be required include importing the TPM Owner Authorization value into the system, calling the Win32_Tpm WMI method for provisioning the TPM and specifying TRUE for either 'ForceClear_Allowed' or 'PhysicalPresencePrompts_Allowed' (as indicated by the value returned in the Additional Information), or enabling the TPM in the system BIOS.)"},
		0x80284014: {"TBS_E_PPI_FUNCTION_UNSUPPORTED", "The Physical Presence Interface of this firmware does not support the requested method."},
		0x80284015: {"TBS_E_OWNERAUTH_NOT_FOUND", "The requested TPM OwnerAuth value was not found."},
		0x80284016: {"TBS_E_PROVISIONING_INCOMPLETE", "The TPM provisioning did not complete. For more information on completing the provisioning, call the Win32_Tpm WMI method for provisioning the TPM ('Provision') and check the returned Information."},
		0x80290100: {"TPMAPI_E_INVALID_STATE", "The command buffer is not in the correct state."},
		0x80290101: {"TPMAPI_E_NOT_ENOUGH_DATA", "The command buffer does not contain enough data to satisfy the request."},
		0x80290102: {"TPMAPI_E_TOO_MUCH_DATA", "The command buffer cannot contain any more data."},
		0x80290103: {"TPMAPI_E_INVALID_OUTPUT_POINTER", "One or more output parameters was NULL or invalid."},
		0x80290104: {"TPMAPI_E_INVALID_PARAMETER", "One or more input parameters is invalid."},
		0x80290105: {"TPMAPI_E_OUT_OF_MEMORY", "Not enough memory was available to satisfy the request."},
		0x80290106: {"TPMAPI_E_BUFFER_TOO_SMALL", "The specified buffer was too small."},
		0x80290107: {"TPMAPI_E_INTERNAL_ERROR", "An internal error was detected."},
		0x80290108: {"TPMAPI_E_ACCESS_DENIED", "The caller does not have the appropriate rights to perform the requested operation."},
		0x80290109: {"TPMAPI_E_AUTHORIZATION_FAILED", "The specified authorization information was invalid."},
		0x8029010A: {"TPMAPI_E_INVALID_CONTEXT_HANDLE", "The specified context handle was not valid."},
		0x8029010B: {"TPMAPI_E_TBS_COMMUNICATION_ERROR", "An error occurred while communicating with the TBS."},
		0x8029010C: {"TPMAPI_E_TPM_COMMAND_ERROR", "The TPM returned an unexpected result."},
		0x8029010D: {"TPMAPI_E_MESSAGE_TOO_LARGE", "The message was too large for the encoding scheme."},
		0x8029010E: {"TPMAPI_E_INVALID_ENCODING", "The encoding in the blob was not recognized."},
		0x8029010F: {"TPMAPI_E_INVALID_KEY_SIZE", "The key size is not valid."},
		0x80290110: {"TPMAPI_E_ENCRYPTION_FAILED", "The encryption operation failed."},
		0x80290111: {"TPMAPI_E_INVALID_KEY_PARAMS", "The key parameters structure was not valid"},
		0x80290112: {"TPMAPI_E_INVALID_MIGRATION_AUTHORIZATION_BLOB", "The requested supplied data does not appear to be a valid migration authorization blob."},
		0x80290113: {"TPMAPI_E_INVALID_PCR_INDEX", "The specified PCR index was invalid."},
		0x80290114: {"TPMAPI_E_INVALID_DELEGATE_BLOB", "The data given does not appear to be a valid delegate blob."},
		0x80290115: {"TPMAPI_E_INVALID_CONTEXT_PARAMS", "One or more of the specified context parameters was not valid."},
		0x80290116: {"TPMAPI_E_INVALID_KEY_BLOB", "The data given does not appear to be a valid key blob"},
		0x80290117: {"TPMAPI_E_INVALID_PCR_DATA", "The specified PCR data was invalid."},
		0x80290118: {"TPMAPI_E_INVALID_OWNER_AUTH", "The format of the owner auth data was invalid."},
		0x80290119: {"TPMAPI_E_FIPS_RNG_CHECK_FAILED", "The random number generated did not pass FIPS RNG check."},
		0x8029011A: {"TPMAPI_E_EMPTY_TCG_LOG", "The TCG Event Log does not contain any data."},
		0x8029011B: {"TPMAPI_E_INVALID_TCG_LOG_ENTRY", "An entry in the TCG Event Log was invalid."},
		0x8029011C: {"TPMAPI_E_TCG_SEPARATOR_ABSENT", "A TCG Separator was not found."},
		0x8029011D: {"TPMAPI_E_TCG_INVALID_DIGEST_ENTRY", "A digest value in a TCG Log entry did not match hashed data."},
		0x8029011E: {"TPMAPI_E_POLICY_DENIES_OPERATION", "The requested operation was blocked by current TPM policy. Please contact your system administrator for assistance."},
		0x80290200: {"TBSIMP_E_BUFFER_TOO_SMALL", "The specified buffer was too small."},
		0x80290201: {"TBSIMP_E_CLEANUP_FAILED", "The context could not be cleaned up."},
		0x80290202: {"TBSIMP_E_INVALID_CONTEXT_HANDLE", "The specified context handle is invalid."},
		0x80290203: {"TBSIMP_E_INVALID_CONTEXT_PARAM", "An invalid context parameter was specified."},
		0x80290204: {"TBSIMP_E_TPM_ERROR", "An error occurred while communicating with the TPM."},
		0x80290205: {"TBSIMP_E_HASH_BAD_KEY", "No entry with the specified key was found."},
		0x80290206: {"TBSIMP_E_DUPLICATE_VHANDLE", "The specified virtual handle matches a virtual handle already in use."},
		0x80290207: {"TBSIMP_E_INVALID_OUTPUT_POINTER", "The pointer to the returned handle location was NULL or invalid."},
		0x80290208: {"TBSIMP_E_INVALID_PARAMETER", "One or more parameters is invalid"},
		0x80290209: {"TBSIMP_E_RPC_INIT_FAILED", "The RPC subsystem could not be initialized."},
		0x8029020A: {"TBSIMP_E_SCHEDULER_NOT_RUNNING", "The TBS scheduler is not running."},
		0x8029020B: {"TBSIMP_E_COMMAND_CANCELED", "The command was canceled."},
		0x8029020C: {"TBSIMP_E_OUT_OF_MEMORY", "There was not enough memory to fulfill the request."},
		0x8029020D: {"TBSIMP_E_LIST_NO_MORE_ITEMS", "The specified list is empty, or the iteration has reached the end of the list."},
		0x8029020E: {"TBSIMP_E_LIST_NOT_FOUND", "The specified item was not found in the list."},
		0x8029020F: {"TBSIMP_E_NOT_ENOUGH_SPACE", "The TPM does not have enough space to load the requested resource."},
		0x80290210: {"TBSIMP_E_NOT_ENOUGH_TPM_CONTEXTS", "There are too many TPM contexts in use."},
		0x80290211: {"TBSIMP_E_COMMAND_FAILED", "The TPM command failed."},
		0x80290212: {"TBSIMP_E_UNKNOWN_ORDINAL", "The TBS does not recognize the specified ordinal."},
		0x80290213: {"TBSIMP_E_RESOURCE_EXPIRED", "The requested resource is no longer available."},
		0x80290214: {"TBSIMP_E_INVALID_RESOURCE", "The resource type did not match."},
		0x80290215: {"TBSIMP_E_NOTHING_TO_UNLOAD", "No resources can be unloaded."},
		0x80290216: {"TBSIMP_E_HASH_TABLE_FULL", "No new entries can be added to the hash table."},
		0x80290217: {"TBSIMP_E_TOO_MANY_TBS_CONTEXTS", "A new TBS context could not be created because there are too many open contexts."},
		0x80290218: {"TBSIMP_E_TOO_MANY_RESOURCES", "A new virtual resource could not be created because there are too many open virtual resources."},
		0x80290219: {"TBSIMP_E_PPI_NOT_SUPPORTED", "The physical presence interface is not supported."},
		0x8029021A: {"TBSIMP_E_TPM_INCOMPATIBLE", "TBS is not compatible with the version of TPM found on the system."},
		0x8029021B: {"TBSIMP_E_NO_EVENT_LOG", "No TCG event log is available."},
		0x80290300: {"TPM_E_PPI_ACPI_FAILURE", "A general error was detected when attempting to acquire the BIOS's response to a Physical Presence command."},
		0x80290301: {"TPM_E_PPI_USER_ABORT", "The user failed to confirm the TPM operation request."},
		0x80290302: {"TPM_E_PPI_BIOS_FAILURE", "The BIOS failure prevented the successful execution of the requested TPM operation (e.g. invalid TPM operation request, BIOS communication error with the TPM)."},
		0x80290303: {"TPM_E_PPI_NOT_SUPPORTED", "The BIOS does not support the physical presence interface."},
		0x80290304: {"TPM_E_PPI_BLOCKED_IN_BIOS", "The Physical Presence command was blocked by current BIOS settings. The system owner may be able to reconfigure the BIOS settings to allow the command."},
		0x80290400: {"TPM_E_PCP_ERROR_MASK", "This is an error mask to convert Platform Crypto Provider errors to win errors."},
		0x80290401: {"TPM_E_PCP_DEVICE_NOT_READY", "The Platform Crypto Device is currently not ready. It needs to be fully provisioned to be operational."},
		0x80290402: {"TPM_E_PCP_INVALID_HANDLE", "The handle provided to the Platform Crypto Provider is invalid."},
		0x80290403: {"TPM_E_PCP_INVALID_PARAMETER", "A parameter provided to the Platform Crypto Provider is invalid."},
		0x80290404: {"TPM_E_PCP_FLAG_NOT_SUPPORTED", "A provided flag to the Platform Crypto Provider is not supported."},
		0x80290405: {"TPM_E_PCP_NOT_SUPPORTED", "The requested operation is not supported by this Platform Crypto Provider."},
		0x80290406: {"TPM_E_PCP_BUFFER_TOO_SMALL", "The buffer is too small to contain all data. No information has been written to the buffer."},
		0x80290407: {"TPM_E_PCP_INTERNAL_ERROR", "An unexpected internal error has occurred in the Platform Crypto Provider."},
		0x80290408: {"TPM_E_PCP_AUTHENTICATION_FAILED", "The authorization to use a provider object has failed."},
		0x80290409: {"TPM_E_PCP_AUTHENTICATION_IGNORED", "The Platform Crypto Device has ignored the authorization for the provider object, to mitigate against a dictionary attack."},
		0x8029040A: {"TPM_E_PCP_POLICY_NOT_FOUND", "The referenced policy was not found."},
		0x8029040B: {"TPM_E_PCP_PROFILE_NOT_FOUND", "The referenced profile was not found."},
		0x8029040C: {"TPM_E_PCP_VALIDATION_FAILED", "The validation was not successful."},
		0x8029040E: {"TPM_E_PCP_WRONG_PARENT", "An attempt was made to import or load a key under an incorrect storage parent."},
		0x8029040F: {"TPM_E_KEY_NOT_LOADED", "The TPM key is not loaded."},
		0x80290410: {"TPM_E_NO_KEY_CERTIFICATION", "The TPM key certification has not been generated."},
		0x80290411: {"TPM_E_KEY_NOT_FINALIZED", "The TPM key is not yet finalized."},
		0x80290412: {"TPM_E_ATTESTATION_CHALLENGE_NOT_SET", "The TPM attestation challenge is not set."},
		0x80290413: {"TPM_E_NOT_PCR_BOUND", "The TPM PCR info is not available."},
		0x80290414: {"TPM_E_KEY_ALREADY_FINALIZED", "The TPM key is already finalized."},
		0x80290415: {"TPM_E_KEY_USAGE_POLICY_NOT_SUPPORTED", "The TPM key usage policy is not supported."},
		0x80290416: {"TPM_E_KEY_USAGE_POLICY_INVALID", "The TPM key usage policy is invalid."},
		0x80290417: {"TPM_E_SOFT_KEY_ERROR", "There was a problem with the software key being imported into the TPM."},
		0x80290418: {"TPM_E_KEY_NOT_AUTHENTICATED", "The TPM key is not authenticated."},
		0x80290419: {"TPM_E_PCP_KEY_NOT_AIK", "The TPM key is not an AIK."},
		0x8029041A: {"TPM_E_KEY_NOT_SIGNING_KEY", "The TPM key is not a signing key."},
		0x8029041B: {"TPM_E_LOCKED_OUT", "The TPM is locked out."},
		0x8029041C: {"TPM_E_CLAIM_TYPE_NOT_SUPPORTED", "The claim type requested is not supported."},
		0x8029041D: {"TPM_E_VERSION_NOT_SUPPORTED", "TPM version is not supported."},
		0x8029041E: {"TPM_E_BUFFER_LENGTH_MISMATCH", "The buffer lengths do not match."},
		0x8029041F: {"TPM_E_PCP_IFX_RSA_KEY_CREATION_BLOCKED", "The RSA key creation is blocked on this TPM due to known security vulnerabilities."},
		0x80290420: {"TPM_E_PCP_TICKET_MISSING", "A ticket required to use a key was not provided."},
		0x80290421: {"TPM_E_PCP_RAW_POLICY_NOT_SUPPORTED", "This key has a raw policy so the KSP can't authenticate against it."},
		0x80290422: {"TPM_E_PCP_KEY_HANDLE_INVALIDATED", "The TPM key's handle was unexpectedly invalidated due to a hardware or firmware issue."},
		0x40290423: {"TPM_E_PCP_UNSUPPORTED_PSS_SALT", "The requested salt size for signing with RSAPSS does not match what the TPM uses."},
		0x40290424: {"TPM_E_PCP_PLATFORM_CLAIM_MAY_BE_OUTDATED", "Validation of the platform claim failed."},
		0x40290425: {"TPM_E_PCP_PLATFORM_CLAIM_OUTDATED", "The requested platform claim is for a previous boot."},
		0x40290426: {"TPM_E_PCP_PLATFORM_CLAIM_REBOOT", "The platform claim is for a previous boot, and cannot be created without reboot."},
		0x80290500: {"TPM_E_EXHAUST_ENABLED", "TPM related network operations are blocked as Zero Exhaust mode is enabled on client."},
		0x80290600: {"TPM_E_PROVISIONING_INCOMPLETE", "TPM provisioning did not run to completion."},
		0x80290601: {"TPM_E_INVALID_OWNER_AUTH", "An invalid owner authorization value was specified."},
		0x80290602: {"TPM_E_TOO_MUCH_DATA", "TPM command returned too much data."},
		0x80090009: {"NTE_BAD_FLAGS", ""},
		0x80090026: {"NTE_INVALID_HANDLE", ""},
		0x80090027: {"NTE_INVALID_PARAMETER", ""},
		0x80090029: {"NTE_NOT_SUPPORTED", ""},
	}
)
